--[[
Copyright (C) GtX (Andy), 2022

Author: GtX | Andy
Date: 02.01.2022
Revision: FS22-03

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

ObjectStorageSpawner = {}

ObjectStorageSpawner.MOD_NAME = g_currentModName
ObjectStorageSpawner.MOD_DIR = g_currentModDirectory

ObjectStorageSpawner.RESULT_SUCCESS = 0
ObjectStorageSpawner.RESULT_PART_SUCCESS = 1
ObjectStorageSpawner.RESULT_NO_SPACE = 2
ObjectStorageSpawner.OBJECT_LIMIT_REACHED = 3
ObjectStorageSpawner.STATUS_UPDATE = 4

ObjectStorageSpawner.COLLISION_MASK = CollisionFlag.VEHICLE + CollisionFlag.PLAYER + CollisionFlag.DYNAMIC_OBJECT + CollisionFlag.TREE

local ObjectStorageSpawner_mt = Class(ObjectStorageSpawner)

local function getIsValidPalletType(typeName)
    -- Backup in case of base game changes. All pallets should use the specialisation 'pallet' as this adds the BOOLEAN 'self.isPallet' allowing for easy identification
    return typeName == "pallet" or typeName == "bigBag" or typeName == "treeSaplingPallet"
end

function ObjectStorageSpawner.registerXMLPaths(schema, basePath)
    schema:register(XMLValueType.NODE_INDEX, basePath .. "#startNode", "Start node")
    schema:register(XMLValueType.NODE_INDEX, basePath .. "#widthNode", "Width node")
    schema:register(XMLValueType.NODE_INDEX, basePath .. "#heightNode", "Height node")
    schema:register(XMLValueType.NODE_INDEX, basePath .. "#lengthNode", "Length node")

    schema:register(XMLValueType.INT, basePath .. "#numRows", "The number of rows the area length will be divided into. (Pallet Storage Only)")
    schema:register(XMLValueType.INT, basePath .. "#maxStackHeight", "The maximum stack height to use, the maximum height for bales less than 0.8m is 4 while all others sizes can stack a maximum of 6 (Bale Storage Only)")
end

function ObjectStorageSpawner.new(isServer, isClient, customMt)
    local self = setmetatable({}, customMt or ObjectStorageSpawner_mt)

    self.spawnQueue = {}
    self.freePlaces = {}

    self.enabled = false
    self.currentObjectToSpawn = nil

    self.numSupplied = 0
    self.numRequested = 0

    self.isServer = isServer
    self.isClient = isClient

    self.startX = 0
    self.startY = 0
    self.startZ = 0

    self.length = 2
    self.height = 2
    self.width = 2

    self.rotX = 0
    self.rotY = 0
    self.rotZ = 0

    self.yRot = 0

    return self
end

function ObjectStorageSpawner:load(xmlFile, key, owner, components, i3dMappings)
    if owner == nil then
        return false
    end

    self.owner = owner

    for _, name in ipairs ({"startNode", "widthNode", "heightNode", "lengthNode"}) do
        local node = xmlFile:getValue(key .. "#" .. name, nil, components, i3dMappings)

        if node == nil then
            Logging.xmlError(xmlFile, "Spawn area '%s' not defined for '%s'", name, key)

            return false
        end

        self[name] = node
    end

    local startX, startY, startZ = getWorldTranslation(self.startNode)
    local widthX, widthY, widthZ = getWorldTranslation(self.widthNode)
    local lenghtX, lenghtY, lenghtZ = getWorldTranslation(self.lengthNode)
    local heightX, heightY, heightZ = getWorldTranslation(self.heightNode)

    self.startX = startX
    self.startY = startY
    self.startZ = startZ

    self.length = MathUtil.vector3Length(startX - lenghtX, startY - lenghtY, startZ - lenghtZ)
    self.height = MathUtil.vector3Length(startX - heightX, startY - heightY, startZ - heightZ)
    self.width = MathUtil.vector3Length(startX - widthX, startY - widthY, startZ - widthZ)

    local rotX, rotY, rotZ = getWorldRotation(self.startNode)
    local dx, _, dz = mathEulerRotateVector(rotX, rotY, rotZ, 0, 0, 1)

    self.rotX = rotX
    self.rotY = rotY
    self.rotZ = rotZ

    self.yRot =  math.atan2(dx, dz)

    if owner.palletStorage then
        local numRows = math.max(xmlFile:getValue(key .. "#numRows", 1), 1)

        local rowLength = self.length / numRows
        local rowSpace = rowLength * 0.5

        local linkNode = getParent(self.startNode)

        self.palletSpawnAreas = {}
        self.rowLength = rowLength

        for i = 1, numRows do
            local startNode = createTransformGroup(string.format("area%dStartNode", i))

            link(linkNode, startNode)

            local lx, ly, lz = localToLocal(self.startNode, linkNode, 0, 0, rowSpace)
            local ldx, ldy, ldz = localDirectionToLocal(self.startNode, linkNode, 0, 0, 1)
            local upx, upy, upz = localDirectionToLocal(self.startNode, linkNode, 0, 1, 0)

            setDirection(startNode, ldx, ldy, ldz, upx, upy, upz)
            setTranslation(startNode, lx, ly, lz)

            local endNode = createTransformGroup(string.format("area%dEndNode", i))

            link(startNode, endNode)

            setTranslation(endNode, self.width, 0, 0)

            local palletSpawnArea = PlacementUtil.loadPlaceFromNode(startNode)

            palletSpawnArea.maxWidth = self.width
            palletSpawnArea.maxLength = rowLength
            palletSpawnArea.maxHeight = self.height

            self.palletSpawnAreas[i] = palletSpawnArea

            rowSpace = rowSpace + rowLength
        end
    else
        self.maxStackHeight = xmlFile:getValue(key .. "#maxStackHeight")
    end

    return true
end

function ObjectStorageSpawner:finalizePlacement()
    self.enabled = true

    return true
end

function ObjectStorageSpawner:delete()
    g_currentMission:removeUpdateable(self)
    self.enabled = false
end

function ObjectStorageSpawner:update(dt)
    if #self.spawnQueue > 0 then
        if self.currentObjectToSpawn == nil then
            local nextQueued = self.spawnQueue[1]
            local storageArea = self.storageArea
            local numObjects = storageArea ~= nil and #storageArea.objects or 0

            self.currentObjectToSpawn = nextQueued

            if numObjects > 0 and nextQueued.place ~= nil then
                local storedObject = storageArea.objects[numObjects]
                local place = nextQueued.place

                if nextQueued.isBale or self.owner.baleStorage then
                    local class = g_objectStorageManager:getBaleClassByFilename(storedObject.xmlFilename)
                    local bale = class.new(self.isServer, self.isClient)

                    if bale:loadFromConfigXML(storedObject.xmlFilename, place.x, place.y, place.z, place.xRot, place.yRot, place.zRot) then
                        self:onFinishLoadingBale(bale, storageArea, storedObject, VehicleLoadingUtil.VEHICLE_LOAD_OK)
                    else
                        self:onFinishLoadingBale(bale, storageArea, storedObject, VehicleLoadingUtil.VEHICLE_LOAD_ERROR)
                    end
                else
                    VehicleLoadingUtil.loadVehicle(storedObject.xmlFilename, place, true, 0, Vehicle.PROPERTY_STATE_OWNED, self.farmId, storedObject.configurations, nil, self.onFinishLoadingPallet, self)
                end
            else
                self.spawnQueue = {}

                self:onSpawnObjectsFinished(ObjectStorageSpawner.STATUS_UPDATE)
                g_currentMission:removeUpdateable(self)
            end
        end
    else
        local statusCode = ObjectStorageSpawner.RESULT_SUCCESS

        if self.numSupplied < self.numRequested then
            statusCode = ObjectStorageSpawner.RESULT_PART_SUCCESS
        end

        self:onSpawnObjectsFinished(statusCode)
        g_currentMission:removeUpdateable(self)
    end
end

function ObjectStorageSpawner:onSpawnObjectsFinished(statusCode)
    self.currentObjectToSpawn = nil

    if self.owner ~= nil then
        self.owner:onSpawnObjectsFinished(statusCode, self.numRequested, self.numSupplied, self.connection)
    end

    self.storageArea = nil
    self.connection = nil

    self.spawnActive = false
end

function ObjectStorageSpawner:spawnObjects(numToSpawn, storageArea, offsetX, offsetY, offsetZ, connection)
    if not self.enabled or numToSpawn == nil or storageArea == nil or self.spawnActive then
        return false
    end

    if connection == nil and self.isServer and self.isClient then
        connection = g_server.clientConnections[NetworkNode.LOCAL_STREAM_ID]
    end

    self.numRequested = numToSpawn
    self.numSupplied = 0

    self.storageArea = storageArea
    self.connection = connection

    self.farmId = self.owner:getOwnerFarmId()
    self.spawnActive = true

    local objectType = storageArea.objectType
    local isBale = not objectType.isPallet

    numToSpawn = math.min(math.min(#storageArea.objects, numToSpawn), self:getMaxSpawnPermitted(isBale))

    if numToSpawn > 0 then
        local freePlaces, numFreePlaces = nil, 0

        if self.owner.baleStorage then
            local isRoundbale = objectType.isRoundbale
            local width = objectType.width
            local height = isRoundbale and objectType.diameter or objectType.height
            local length = isRoundbale and objectType.diameter or objectType.length

            freePlaces, numFreePlaces = self:getFreePlaces(width, height, length, offsetX, offsetY, offsetZ, isBale, isRoundbale, numToSpawn)
        else
            freePlaces, numFreePlaces = self:getFreePalletPlaces(storageArea.objects, numToSpawn)
        end

        numFreePlaces = math.min(numFreePlaces, numToSpawn)

        if numFreePlaces > 0 then
            for i = 1, math.min(numFreePlaces, numToSpawn) do
                table.insert(self.spawnQueue, {
                    width = width,
                    height = height,
                    length = length,
                    place = freePlaces[i],
                    isBale = isBale,
                    isRoundbale = isRoundbale
                })
            end

            g_currentMission:addUpdateable(self)

            return true
        end

        self:onSpawnObjectsFinished(ObjectStorageSpawner.RESULT_NO_SPACE)
    else
        self:onSpawnObjectsFinished(ObjectStorageSpawner.OBJECT_LIMIT_REACHED)
    end

    return false
end

function ObjectStorageSpawner:getMaxSpawnPermitted(isBale)
    if self.isServer and g_currentMission.slotSystem ~= nil then
        local objectType = SlotSystem.LIMITED_OBJECT_PALLET

        if isBale then
            objectType = SlotSystem.LIMITED_OBJECT_BALE
        end

        local objectData = g_currentMission.slotSystem.objectLimits[objectType]

        if objectData ~= nil then
            return objectData.limit - #objectData.objects
        end
    end

    return 0
end

function ObjectStorageSpawner:getFreePalletPlaces(objects, maxRequiredPlaces)
    local freePlaces = {}
    local numFreePlaces = 0

    if self.enabled then
        local usedSpawnAreaWidth = {}
        local terrainRootNode = g_currentMission.terrainRootNode

        for i = #objects, 1, -1 do
            local size = g_objectStorageManager:getPalletSizeByFilename(objects[i].xmlFilename)
            local freePlace = self:getFreePalletPlace(size, usedSpawnAreaWidth, terrainRootNode)

            if freePlace ~= nil then
                numFreePlaces = numFreePlaces + 1

                freePlaces[numFreePlaces] = freePlace

                if numFreePlaces == maxRequiredPlaces then
                    return freePlaces, numFreePlaces
                end
            end
        end
    end

    return freePlaces, numFreePlaces
end

function ObjectStorageSpawner:getFreePalletPlace(size, usedSpawnAreaWidth, terrainRootNode)
    for _, spawnArea in ipairs(self.palletSpawnAreas) do
        local usedWidth = usedSpawnAreaWidth[spawnArea] or 0
        local halfWidth = size.width * 0.5

        for width = usedWidth + halfWidth, spawnArea.width - halfWidth, halfWidth do
            local x = spawnArea.startX + width * spawnArea.dirX
            local y = spawnArea.startY + width * spawnArea.dirY
            local z = spawnArea.startZ + width * spawnArea.dirZ
            local terrainHeight = getTerrainHeightAtWorldPos(terrainRootNode, x, y, z)

            y = math.max(terrainHeight + 0.5, y)

            self.collisionObjectId = 0

            overlapBox(x, y, z, spawnArea.rotX, spawnArea.rotY, spawnArea.rotZ, size.width * 0.5, spawnArea.maxHeight * 0.5, size.length * 0.5, "collisionTestCallback", self, ObjectStorageSpawner.COLLISION_MASK, true, false, true)

            if self.collisionObjectId == 0 then
                local yRot = MathUtil.getYRotationFromDirection(spawnArea.dirPerpX, spawnArea.dirPerpZ)
                local yOffset = size.height - (math.max(terrainHeight + spawnArea.yOffset, y) - terrainHeight)

                usedSpawnAreaWidth[spawnArea] = width + halfWidth

                return {
                    x = x - size.widthOffset * spawnArea.dirX - size.lengthOffset * spawnArea.dirPerpX,
                    y = y - size.widthOffset * spawnArea.dirY - size.lengthOffset * spawnArea.dirPerpY,
                    z = z - size.widthOffset * spawnArea.dirZ - size.lengthOffset * spawnArea.dirPerpZ,
                    xRot = 0,
                    yRot = yRot,
                    zRot = 0,
                    ex = size.width * 0.5,
                    ey = size.height * 0.5,
                    ez = size.length * 0.5,
                    yOffset = yOffset * 0.5
                }
            end
        end
    end

    return nil
end

function ObjectStorageSpawner:getFreePlaces(sizeX, sizeY, sizeZ, offsetX, offsetY, offsetZ, isBale, isRoundbale, maxRequiredPlaces)
    local freePlaces = {}
    local numFreePlaces = 0

    if self.enabled then
        local length = sizeX
        local height = sizeY
        local width = sizeZ

        local extentX = sizeX * 0.5
        local extentY = sizeY * 0.5
        local extentZ = sizeZ * 0.5

        local rotX = 0
        local rotY = self.yRot
        local rotZ = 0

        local maxStackHeight = 1

        offsetX = 0.1
        offsetY = 0.025
        offsetZ = 0.1

        if isBale then
            maxStackHeight = math.min(self.maxStackHeight or 6, 6)

            if isRoundbale then
                length = sizeZ
                height = sizeX
                width = sizeY

                extentX = sizeZ * 0.5
                extentZ = sizeX * 0.5

                rotX = rotX - (math.pi / 2)
            else
                length = sizeZ
                width = sizeX

                if height < 0.8 then
                    maxStackHeight = math.min(self.maxStackHeight or 4, 4)

                    offsetX = 0.05
                    offsetY = 0.02
                    offsetZ = 0.05
                end
            end
        else
            rotY = rotY + (math.pi / 2)
        end

        length = length + offsetX
        height = height + offsetY
        width = width + offsetZ

        local lengthHalf = length * 0.5
        local heightHalf = height * 0.5
        local widthHalf = width * 0.5

        local startNode = self.startNode

        local areaLength = self.length
        local maxLength = math.floor(areaLength / length)
        local placeLength = (areaLength - (length * maxLength)) / (maxLength + 1)
        local stepLength = placeLength + length

        local areaHeight = math.min(self.height, height * maxStackHeight)

        local areaWidth = self.width
        local maxWidth = math.floor(areaWidth / width)
        local placeWidth = (areaWidth - (width * maxWidth)) / (maxWidth + 1)
        local stepWidth = placeWidth + width

        local terrainRootNode = g_currentMission.terrainRootNode

        for dz = lengthHalf + placeLength, areaLength - lengthHalf, stepLength do
            for dx = widthHalf + placeWidth, areaWidth - widthHalf, stepWidth do
                local lastSizeX, lastSizeY, lastSizeZ, yOffset, invalidPos = nil, nil, nil, 0, false

                for dy = heightHalf, areaHeight - heightHalf, height do
                    local x, y, z = localToWorld(startNode, dx, dy, dz)
                    local canSpawn = true

                    if yOffset == 0 then
                        yOffset = 0.01

                        local terrainY = getTerrainHeightAtWorldPos(terrainRootNode, x, 0, z)

                        if terrainY > y - heightHalf then
                            yOffset = yOffset + terrainY - (y - heightHalf)
                        end
                    end

                    y = y + yOffset

                    self.collisionObjectId = 0

                    overlapBox(x, y, z, rotX, rotY, rotZ, extentX, extentY, extentZ, "collisionTestCallback", self, ObjectStorageSpawner.COLLISION_MASK, true, false, true)

                    if self.collisionObjectId ~= 0 then
                        if lastSizeX == nil then
                            local object = g_currentMission:getNodeObject(self.collisionObjectId)

                            if object ~= nil and object.isa ~= nil then
                                if isBale then
                                    if object:isa(Bale) then
                                        local bx, by, bz = getWorldTranslation(object.nodeId)
                                        local maxOffset = 0.3

                                        lastSizeX = MathUtil.round(object.width, 2)

                                        if object.isRoundbale then
                                            lastSizeY = MathUtil.round(object.diameter, 2)
                                            lastSizeZ = lastSizeY

                                            maxOffset = sizeY / 4
                                        else
                                            lastSizeY = MathUtil.round(object.height, 2)
                                            lastSizeZ = MathUtil.round(object.length, 2)

                                            maxOffset = sizeX / 4
                                        end

                                        if MathUtil.vector2Length(bx - x, bz - z) > maxOffset then
                                            invalidPos = true
                                        end
                                    end
                                elseif object:isa(Vehicle) then
                                    if object.isPallet or getIsValidPalletType(object.typeName) then
                                        lastSizeX = MathUtil.round(object.size.width, 2)
                                        lastSizeY = MathUtil.round(object.size.height, 2)
                                        lastSizeZ = MathUtil.round(object.size.length, 2)
                                    elseif object.spec_wheels == nil and object.spec_enterable == nil then
                                        -- Mods with special pallet specs for some reason
                                        for _, spec in pairs(object.specializations) do
                                            if spec == Pallet or spec == BigBag or spec == TreeSaplingPallet then
                                                lastSizeX = MathUtil.round(object.size.width, 2)
                                                lastSizeY = MathUtil.round(object.size.height, 2)
                                                lastSizeZ = MathUtil.round(object.size.length, 2)

                                                break
                                            end
                                        end
                                    end
                                end
                            end
                        end

                        canSpawn = false
                    else
                        if invalidPos or lastSizeX ~= nil and (sizeX ~= lastSizeX or sizeY ~= lastSizeY or sizeZ ~= lastSizeZ) then
                            canSpawn = false
                        end
                    end

                    if canSpawn then
                        numFreePlaces = numFreePlaces + 1

                        freePlaces[numFreePlaces] = {
                            x = x,
                            y = y,
                            z = z,
                            xRot = rotX,
                            yRot = rotY,
                            zRot = rotZ,
                            ex = extentX,
                            ey = extentY,
                            ez = extentZ,
                            yOffset = 0
                        }

                        if numFreePlaces == maxRequiredPlaces then
                            return freePlaces, numFreePlaces
                        end
                    end
                end
            end
        end
    end

    return freePlaces, numFreePlaces
end

function ObjectStorageSpawner:collisionTestCallback(transformId)
    if transformId ~= 0 and not Utils.getNoNil(getUserAttribute(transformId, "allowObjectSpawning"), false) then
        self.collisionObjectId = transformId

        return false
    end

    return true
end

function ObjectStorageSpawner:getAreaBlocked()
    if self.enabled then
        local extentX = self.width * 0.5
        local extentY = self.height * 0.5
        local extentZ = self.length * 0.5

        local x, y, z = localToWorld(self.startNode, extentX, extentY, extentZ)

        self.objectInArea = false
        self.objectInAreaName = ""

        overlapBox(x, y, z, self.rotX, self.rotY, self.rotZ, extentX, extentY, extentZ, "areaCollisionTestCallback", self, ObjectStorageSpawner.COLLISION_MASK, true, false, true)

        return self.objectInArea, self.objectInAreaName
    end

    return true, ""
end

function ObjectStorageSpawner:areaCollisionTestCallback(transformId)
    if transformId ~= 0 and not Utils.getNoNil(getUserAttribute(transformId, "allowObjectSpawning"), false) then
        local object = g_currentMission:getNodeObject(transformId)

        if object ~= nil then
            if object.spec_enterable ~= nil or object.spec_wheels ~= nil then
                self.objectInArea = true
                self.objectInAreaName = object:getFullName()
            end
        else
            local player = g_currentMission.players[transformId]

            if player ~= nil then
                local playerNickname = ""

                if player == g_currentMission.player then
                    playerNickname = g_currentMission.playerNickname
                else
                    local user = g_currentMission.userManager:getUserByUserId(player.userId)

                    if user ~= nil then
                        playerNickname = user:getNickname()
                    end
                end

                if playerNickname == "" then
                    playerNickname = g_i18n:getText("ui_playerCharacter")
                end

                self.objectInArea = true
                self.objectInAreaName = playerNickname
            elseif getHasClassId(transformId, ClassIds.MESH_SPLIT_SHAPE) and getSplitType(transformId) ~= 0 then
                self.objectInArea = true
                self.objectInAreaName = g_i18n:getText("infohud_wood")
            end
        end
    end

    return not self.objectInArea
end

function ObjectStorageSpawner:onFinishLoadingBale(bale, storageArea, attributes, loadState)
    if loadState == VehicleLoadingUtil.VEHICLE_LOAD_OK then
        bale:applyBaleAttributes(attributes)
        bale:register()

        self.owner:removeFromStorage(storageArea)
        self.numSupplied = self.numSupplied + 1
    end

    self.currentObjectToSpawn = nil

    table.remove(self.spawnQueue, 1)
end

function ObjectStorageSpawner:onFinishLoadingPallet(pallet, loadState)
    if loadState == VehicleLoadingUtil.VEHICLE_LOAD_OK and pallet ~= nil then
        local storageArea = self.storageArea
        local numObjects = storageArea ~= nil and #storageArea.objects or 0
        local storedObject = storageArea.objects[numObjects]
        local fillUnit = pallet.spec_fillUnit.fillUnits[1]

        if storedObject ~= nil and storageArea.fillTypeIndex ~= nil and fillUnit ~= nil then
            if pallet.emptyAllFillUnits ~= nil then
                pallet:emptyAllFillUnits(true)
            end

            -- Make sure that the capacity is high enough in case pallet has changed due to fillType changes
            local capacity = math.max(storedObject.capacity or 0, storedObject.fillLevel)

            if fillUnit.capacity < capacity then
                fillUnit.capacity = capacity
                pallet.objectStorageCapacity = capacity
            end

            pallet:addFillUnitFillLevel(self.farmId, 1, storedObject.fillLevel, storageArea.fillTypeIndex, ToolType.UNDEFINED)

            self.owner:removeFromStorage(storageArea)
            self.numSupplied = self.numSupplied + 1
        else
            g_currentMission:removeVehicle(pallet) -- ??
        end
    end

    self.currentObjectToSpawn = nil

    table.remove(self.spawnQueue, 1)
end

function ObjectStorageSpawner:drawAreas(sizeX, sizeY, sizeZ, isBale, isRoundbale, storageArea)
    if self.enabled then
        local extentX = self.width * 0.5
        local extentY = self.height * 0.5
        local extentZ = self.length * 0.5

        local freePlaces, numFreePlaces, areaIndex = nil, 0, 0
        local isBlocked, objectName = self:getAreaBlocked()
        local x, y, z = localToWorld(self.startNode, extentX, extentY, extentZ)

        if isBlocked then
            if objectName ~= nil and objectName ~= "" then
                Utils.renderTextAtWorldPosition(x, y, z, string.format("Area blocked by: %s", objectName), getCorrectTextSize(0.02), 0)
            end

            DebugUtil.drawOverlapBox(x, y, z, self.rotX, self.rotY, self.rotZ, extentX, extentY, extentZ, 1, 0, 0)
        else
            if self.owner.baleStorage or storageArea == nil then
                freePlaces, numFreePlaces = self:getFreePlaces(sizeX, sizeY, sizeZ, nil, nil, nil, isBale, isRoundbale)
            else
                if storageArea.numObjects < 1 then
                    local tempObjects = {}

                    for i = 1, math.floor(self.width / sizeX) do
                        tempObjects[i] = {
                            xmlFilename = storageArea.objectType.xmlFilename
                        }
                    end

                    freePlaces, numFreePlaces = self:getFreePalletPlaces(tempObjects)
                else
                    freePlaces, numFreePlaces = self:getFreePalletPlaces(storageArea.objects)
                end

                areaIndex = storageArea.index
            end

            for _, place in pairs (freePlaces) do
                DebugUtil.drawOverlapBox(place.x, place.y + place.yOffset, place.z, place.xRot, place.yRot, place.zRot, place.ex, place.ey, place.ez, 0, 1, 0)
            end

            DebugUtil.drawOverlapBox(x, y, z, self.rotX, self.rotY, self.rotZ, extentX, extentY, extentZ, 0, 0, 1)
        end

        if areaIndex > 0 then
            Utils.renderTextAtWorldPosition(x, y + extentY + 1, z, string.format("Area Index: %d | Free Spaces = %d", areaIndex, numFreePlaces), getCorrectTextSize(0.02), 0)
        else
            Utils.renderTextAtWorldPosition(x, y + extentY + 1, z, string.format("%.2f x %.2f x %.2f | isBale = %s | isRoundbale = %s | Free Spaces = %d", sizeX, sizeY, sizeZ, isBale, isRoundbale, numFreePlaces), getCorrectTextSize(0.02), 0)
        end
    end
end

function ObjectStorageSpawner:getCanSpawnSize(width, height, length)
    if self.owner.palletStorage then
        if self.width < width or self.height < height or self.rowLength < length then
            return false
        end
    else
        if self.width < width or self.height < height or self.length < length then
            return false
        end
    end

    return true
end
