--[[
Copyright (C) GtX (Andy), 2019

Author: GtX | Andy
Date: 25.08.2019
Revision: FS22-02

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

BaleObjectStorage = {}

BaleObjectStorage.MOD_NAME = g_currentModName
BaleObjectStorage.MOD_DIR = g_currentModDirectory

local BaleObjectStorage_mt = Class(BaleObjectStorage, ObjectStorage)

InitObjectClass(BaleObjectStorage, "BaleObjectStorage")

function BaleObjectStorage.registerXMLPaths(schema, basePath)
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas#minBaleDiameter", "Bale min diameter accepted by all storage areas", 0.1)
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas#maxBaleDiameter", "Bale max diameter accepted by all storage areas", 10)
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas#minBaleLength", "Bale min length accepted by all storage areas", 0.1)
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas#maxBaleLength", "Bale max length accepted by all storage areas", 10)

    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas.storageArea(?).acceptedBaleSizes#minDiameter", "Bale min diameter accepted by storage area (Optional)", "minBaleDiameter")
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas.storageArea(?).acceptedBaleSizes#maxDiameter", "Bale max diameter accepted by storage area (Optional)", "maxBaleDiameter")
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas.storageArea(?).acceptedBaleSizes#minLength", "Bale min length accepted by storage area (Optional)", "minBaleLength")
    schema:register(XMLValueType.FLOAT, basePath .. ".storageAreas.storageArea(?).acceptedBaleSizes#maxLength", "Bale max length accepted by storage area (Optional)", "maxBaleLength")

    schema:register(XMLValueType.STRING, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#excludedFillTypes", "Bale fill types that will be ignored when 'sharedVisibilityNodes.visibilityNodes(?)#fillTypes' is not used")
    schema:register(XMLValueType.BOOL, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#isRoundbale", "Bale is a roundbale", true)
    schema:register(XMLValueType.FLOAT, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#width", "Bale Width", 0)
    schema:register(XMLValueType.FLOAT, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#height", "Bale Height", 0)
    schema:register(XMLValueType.FLOAT, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#length", "Bale Length", 0)
    schema:register(XMLValueType.FLOAT, basePath .. ".sharedVisibilityNodes.visibilityNodes(?)#diameter", "Bale Diameter", 0)
end

function BaleObjectStorage.registerSavegameXMLPaths(schema, basePath)
    schema:register(XMLValueType.STRING, basePath .. ".area(?).object(?)#filename", "Path to bale xml file")
    schema:register(XMLValueType.FLOAT, basePath .. ".area(?).object(?)#valueScale", "Bale value scale")
    schema:register(XMLValueType.FLOAT, basePath .. ".area(?).object(?)#fillLevel", "Current bale fill level")
    schema:register(XMLValueType.FLOAT, basePath .. ".area(?).object(?)#capacity", "Bale capacity if it is not standard")
    schema:register(XMLValueType.STRING, basePath .. ".area(?).object(?)#fillType", "Current bale fill type")
    schema:register(XMLValueType.BOOL, basePath .. ".area(?).object(?)#isMissionBale", "Bale was produced in mission context", false)
    schema:register(XMLValueType.FLOAT, basePath .. ".area(?).object(?)#wrappingState", "Current wrapping state")
    schema:register(XMLValueType.VECTOR_4, basePath .. ".area(?).object(?)#wrappingColor", "Wrapping color")
    schema:register(XMLValueType.BOOL, basePath .. ".area(?).object(?)#isFermenting", "Bale is fermenting")
    schema:register(XMLValueType.FLOAT, basePath .. ".area(?).object(?)#fermentationTime", "Current fermentation time")
    schema:register(XMLValueType.FLOAT, basePath .. ".area(?).object(?)#fermentationMaxTime", "Fermentation max time")
    schema:register(XMLValueType.STRING, basePath .. ".area(?).object(?).textures#wrapDiffuse", "Current wrap diffuse file")
    schema:register(XMLValueType.STRING, basePath .. ".area(?).object(?).textures#wrapNormal", "Current wrap normal file")
    schema:register(XMLValueType.INT, basePath .. ".area(?).object(?)#index", "Stored index")
end

function BaleObjectStorage.new(isServer, isClient, baseDirectory, customEnvironment, customMt)
    local self = ObjectStorage.new(isServer, isClient, baseDirectory, customEnvironment, customMt or BaleObjectStorage_mt)

    self.name = string.format("%s %s", g_i18n:getText("infohud_bale"), g_i18n:getText("statistic_storage"))

    self.baleStorage = true
    self.palletStorage = false

    self.balesInTrigger = {}
    self.processBalesInTrigger = true

    self.infoTableFermentingText = "  - " .. g_i18n:getText("info_fermenting")
    self.infoTableBaleText = g_i18n:getText("category_bales")

    return self
end

function BaleObjectStorage:load(xmlFile, key, components, i3dMappings)
    if not BaleObjectStorage:superClass().load(self, xmlFile, key, components, i3dMappings) then
        return false
    end

    local availableObjectTypes = g_objectStorageManager:getAvailableBaleObjectTypes()
    local availableFillTypes = g_objectStorageManager:getAvailableBaleFillTypes()

    if (availableObjectTypes == nil or #availableObjectTypes == 0) or (availableFillTypes == nil or table.size(availableFillTypes) == 0) then
        Logging.xmlError(xmlFile, "No valid object types defined!")

        return false
    end

    local acceptedFillTypes = nil
    local fillTypeNames = xmlFile:getValue(key .. ".storageAreas#fillTypes")

    if fillTypeNames ~= nil then
        fillTypeNames = string.split(fillTypeNames, " ")

        if #fillTypeNames > 0 then
            acceptedFillTypes = {}

            for _, fillTypeName in pairs(fillTypeNames) do
                local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)

                if fillTypeIndex ~= nil and availableFillTypes[fillTypeIndex] then
                    acceptedFillTypes[fillTypeIndex] = true
                end
            end
        end
    end

    if acceptedFillTypes == nil then
        acceptedFillTypes = {}

        for fillTypeIndex, _ in pairs(availableFillTypes) do
            acceptedFillTypes[fillTypeIndex] = true
        end
    end

    local minDiameter = MathUtil.round(xmlFile:getValue(key .. ".storageAreas#minBaleDiameter", 0.1), 2)
    local maxDiameter = MathUtil.round(xmlFile:getValue(key .. ".storageAreas#maxBaleDiameter", 10), 2)
    local minLength = MathUtil.round(xmlFile:getValue(key .. ".storageAreas#minBaleLength", 0.1), 2)
    local maxLength = MathUtil.round(xmlFile:getValue(key .. ".storageAreas#maxBaleLength", 10), 2)

    local sortedAcceptedTypes = g_objectStorageManager:getSortedAcceptedBaleTypes(acceptedFillTypes, minDiameter, maxDiameter, minLength, maxLength)

    if #sortedAcceptedTypes == 0 then
        Logging.xmlError(xmlFile, "Object storage (%s) does not support any available fill types!", self.owningPlaceable.configFileName)

        return false
    end

    self.availableObjectTypes = availableObjectTypes
    self.acceptedFillTypes = acceptedFillTypes
    self.sortedAcceptedTypes = sortedAcceptedTypes

    local function getAreaAcceptedTypes(areaKey, areaIndex)
        if xmlFile:hasProperty(areaKey .. ".acceptedBaleSizes") then
            local minAcceptedDiameter = math.max(MathUtil.round(xmlFile:getValue(areaKey .. ".acceptedBaleSizes#minDiameter", minDiameter), 2), minDiameter)
            local maxAcceptedDiameter = math.min(MathUtil.round(xmlFile:getValue(areaKey .. ".acceptedBaleSizes#maxDiameter", maxDiameter), 2), maxDiameter)
            local minAcceptedLength = math.max(MathUtil.round(xmlFile:getValue(areaKey .. ".acceptedBaleSizes#minLength", minLength), 2), minLength)
            local maxAcceptedLength = math.min(MathUtil.round(xmlFile:getValue(areaKey .. ".acceptedBaleSizes#maxLength", maxLength), 2), maxLength)

            if minAcceptedDiameter > minDiameter or maxAcceptedDiameter < maxDiameter or minAcceptedLength > minLength or maxAcceptedLength < maxLength then
                local acceptedTypes = {}
                local numAcceptedTypes = 0

                for _, acceptedType in ipairs (self.sortedAcceptedTypes) do
                    if ObjectStorageManager.getBaleSizeInRange(acceptedType.objectType, minAcceptedDiameter, maxAcceptedDiameter, minAcceptedLength, maxAcceptedLength) then
                        numAcceptedTypes = numAcceptedTypes + 1
                        acceptedTypes[numAcceptedTypes] = acceptedType
                    end
                end

                if numAcceptedTypes > 0 and numAcceptedTypes ~= #self.sortedAcceptedTypes then
                    return acceptedTypes, numAcceptedTypes
                end
            end
        end

        return nil, 0
    end

    self:loadSharedVisibilityNodes(xmlFile, key, acceptedFillTypes, true)
    self:loadStorageAreas(xmlFile, key, components, i3dMappings, acceptedFillTypes, sortedAcceptedTypes, getAreaAcceptedTypes)

    self.fermentingDirtyFlag = self:getNextDirtyFlag()

    return true
end

function BaleObjectStorage:getDisplayFunction(displayType)
    if displayType == "BALESIZE" then
        return function(display, _, _, _, storageArea)
            local size = 0

            if storageArea ~= nil and storageArea.objectType ~= nil then
                size = storageArea.objectType.isRoundbale and storageArea.objectType.diameter or storageArea.objectType.length
                size = (size or 0) * 100
            end

            if size ~= display.lastValue then
                local int, floatPart = math.modf(size)
                local value = string.format(display.formatStr, int, math.abs(math.floor((floatPart + 1e-06) * 10 ^ display.formatPrecision)))

                display.fontMaterial:updateCharacterLine(display.characterLine, value)
                display.lastValue = size
            end
        end, true
    end

    return BaleObjectStorage:superClass().getDisplayFunction(self, displayType)
end

function BaleObjectStorage:delete()
    g_objectStorageManager:removeFermentingStorage(self)

    self.processBalesInTrigger = false
    self.balesInTrigger = {}

    BaleObjectStorage:superClass().delete(self)
end

function BaleObjectStorage:readStream(streamId, connection)
    BaleObjectStorage:superClass().readStream(self, streamId, connection)

    if connection:getIsServer() then
        for _, storageArea in ipairs (self.indexedStorageAreas) do
            storageArea.numObjects = streamReadUIntN(streamId, ObjectStorage.OBJECTS_SEND_NUM_BITS)
            storageArea.fillTypeIndex = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS)

            local objectTypeIndex = streamReadUIntN(streamId, ObjectStorage.TYPE_SEND_NUM_BITS)
            storageArea.objectType = self:getObjectType(objectTypeIndex, storageArea.fillTypeIndex)

            self:setVisibilityNodes(storageArea, self.onStorageObjectTypeChanged, self)

            if storageArea.numObjects > 0 then
                for i = 1, storageArea.numObjects do
                    local attributes = {
                        isFermenting = false,
                        wrappingState = 0
                    }

                    attributes.fillLevel = streamReadFloat32(streamId)
                    attributes.capacity = streamReadFloat32(streamId)

                    if streamReadBool(streamId) then
                        attributes.wrappingState = streamReadUInt8(streamId) / 255

                        local r = streamReadFloat32(streamId)
                        local g = streamReadFloat32(streamId)
                        local b = streamReadFloat32(streamId)
                        local a = streamReadFloat32(streamId)

                        attributes.wrappingColor = {r, g, b, a}
                    end

                    if streamReadBool(streamId) then
                        attributes.fermentingPercentage = streamReadUInt8(streamId) / 255
                        attributes.isFermenting = true
                    end

                    storageArea.objects[i] = attributes
                end
            end

            self:raiseStorageUpdate(storageArea)
        end
    end
end

function BaleObjectStorage:writeStream(streamId, connection)
    BaleObjectStorage:superClass().writeStream(self, streamId, connection)

    if not connection:getIsServer() then
        for _, storageArea in ipairs (self.indexedStorageAreas) do
            storageArea.numObjects = #storageArea.objects

            streamWriteUIntN(streamId, storageArea.numObjects, ObjectStorage.OBJECTS_SEND_NUM_BITS)
            streamWriteUIntN(streamId, storageArea.fillTypeIndex, FillTypeManager.SEND_NUM_BITS)

            local objectTypeIndex = storageArea.objectType and storageArea.objectType.index or 0
            streamWriteUIntN(streamId, objectTypeIndex, ObjectStorage.TYPE_SEND_NUM_BITS)

            if storageArea.numObjects > 0 then
                for i = 1, storageArea.numObjects do
                    local attributes = storageArea.objects[i]
                    local fillLevel = attributes.fillLevel or 0

                    streamWriteFloat32(streamId, fillLevel)
                    streamWriteFloat32(streamId, attributes.capacity or fillLevel)

                    if streamWriteBool(streamId, attributes.wrappingState > 0) then
                        streamWriteUInt8(streamId, MathUtil.clamp(attributes.wrappingState * 255, 0, 255))

                        streamWriteFloat32(streamId, attributes.wrappingColor[1] or 1)
                        streamWriteFloat32(streamId, attributes.wrappingColor[2] or 1)
                        streamWriteFloat32(streamId, attributes.wrappingColor[3] or 1)
                        streamWriteFloat32(streamId, attributes.wrappingColor[4] or 1)
                    end

                    if streamWriteBool(streamId, attributes.isFermenting) then
                        streamWriteUInt8(streamId, MathUtil.clamp(attributes.fermentingPercentage * 255, 0, 255))
                    end
                end
            end
        end
    end
end

function BaleObjectStorage:readUpdateStream(streamId, timestamp, connection)
    BaleObjectStorage:superClass().readUpdateStream(self, streamId, timestamp, connection)

    if connection:getIsServer() then
        if streamReadBool(streamId) then
            for _, storageArea in ipairs (self.indexedStorageAreas) do
                local fermentingBales = {}
                local numFermentingBales = streamReadUIntN(streamId, ObjectStorage.OBJECTS_SEND_NUM_BITS)

                if numFermentingBales > 0 then
                    for i = 1, numFermentingBales do
                        local index = streamReadUIntN(streamId, ObjectStorage.OBJECTS_SEND_NUM_BITS)
                        fermentingBales[index] = streamReadUInt8(streamId) / 255
                    end
                end

                for i, attributes in ipairs (storageArea.objects) do
                    local fermentingPercentage = fermentingBales[i]

                    if fermentingPercentage ~= nil then
                        attributes.fermentingPercentage = fermentingPercentage
                        attributes.isFermenting = true
                    else
                        attributes.isFermenting = false
                        attributes.fermentingPercentage = nil
                    end
                end
            end
        end
    end
end

function BaleObjectStorage:writeUpdateStream(streamId, connection, dirtyMask)
    BaleObjectStorage:superClass().writeUpdateStream(self, streamId, connection, dirtyMask)

    if not connection:getIsServer() then
        if streamWriteBool(streamId, bitAND(dirtyMask, self.fermentingDirtyFlag) ~= 0) then
            for _, storageArea in ipairs (self.indexedStorageAreas) do
                local fermentingBales = self:getFermentingBales(storageArea)
                local numFermentingBales = fermentingBales ~= nil and #fermentingBales or 0

                streamWriteUIntN(streamId, numFermentingBales, ObjectStorage.OBJECTS_SEND_NUM_BITS)

                if numFermentingBales > 0 then
                    for i = 1, numFermentingBales do
                        streamWriteUIntN(streamId, fermentingBales[i].index, ObjectStorage.OBJECTS_SEND_NUM_BITS)
                        streamWriteUInt8(streamId, MathUtil.clamp(fermentingBales[i].fermentingPercentage * 255, 0, 255))
                    end
                end
            end
        end
    end
end

function BaleObjectStorage:update(dt)
    BaleObjectStorage:superClass().update(self, dt)

    if self.isServer and self.processBalesInTrigger then
        for index, bale in ipairs(self.balesInTrigger) do
            if self.inputTriggerState and bale ~= nil and bale.nodeId ~= 0 and bale:getFillLevel() > 0.01 then
                if bale:getCanBeSold() and bale.dynamicMountJointIndex == nil then
                    local fillTypeIndex = bale:getFillType()
                    local isFermenting = bale.isFermenting
                    local fermentingMaxTime = 1

                    if isFermenting then
                        local fillTypeInfo = bale:getFillTypeInfo(fillTypeIndex)

                        if fillTypeInfo ~= nil and fillTypeInfo.fermenting ~= nil then
                            fillTypeIndex = fillTypeInfo.fermenting.outputFillTypeIndex
                            fermentingMaxTime = fillTypeInfo.fermenting.time or 1
                        end
                    end

                    local storageArea = self:getValidStorageArea(fillTypeIndex, self:getBaleObjectTypeIndex(bale, fillTypeIndex), true)

                    if storageArea ~= nil then
                        local attributes = bale:getBaleAttributes()

                        if attributes.fillLevel > 0 then
                            attributes.capacity = bale:getCapacity() or attributes.fillLevel

                            if isFermenting then
                                attributes.fermentingPercentage = bale.fermentingPercentage or 0
                                attributes.fermentingMaxTime = fermentingMaxTime * ObjectStorageManager.FERMENTING_FACTOR
                            end

                            if self:addToStorage(storageArea, attributes) then
                                bale:delete()
                            end
                        end
                    end

                    table.remove(self.balesInTrigger, index)
                end
            else
                table.remove(self.balesInTrigger, index)
            end

            if #self.balesInTrigger > 0 then
                self:raiseActive()
            end
        end
    end
end

function BaleObjectStorage:loadFromXMLFile(xmlFile, key)
    BaleObjectStorage:superClass().loadFromXMLFile(self, xmlFile, key)

    local farmId = self.owningPlaceable:getOwnerFarmId()

    local addFermentingStorage = false
    local economicDifficulty = g_currentMission.missionInfo.economicDifficulty

    for areaIndex, storageArea in ipairs (self.indexedStorageAreas) do
        local areaKey = string.format("%s.area(%d)", key, areaIndex - 1)

        local defaultFilename = xmlFile:getValue(areaKey .. "#defaultFilename")
        local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(xmlFile:getValue(areaKey .. "#fillType"), "INVALID")

        if defaultFilename ~= nil and self.acceptedFillTypes[fillTypeIndex] then
            local defaultAttributes, bale = g_objectStorageManager:getBaleAttributesByFilename(NetworkUtil.convertFromNetworkFilename(defaultFilename), fillTypeIndex, farmId)

            if defaultAttributes ~= nil then
                local typeIndex = self:getBaleObjectTypeIndex(bale)
                local validToLoad = false

                if typeIndex ~= 0 then
                    for i = 1, #self.sortedAcceptedTypes do
                        if self.sortedAcceptedTypes[i].objectTypeIndex == typeIndex then
                            validToLoad = true

                            break
                        end
                    end
                end

                if validToLoad and self:setStorageObjectType(areaIndex, typeIndex, fillTypeIndex, true) then
                    local numObjects = MathUtil.clamp(xmlFile:getValue(areaKey .. "#numObjects", storageArea.numObjects), 0, storageArea.maxObjects)

                    storageArea.loadedFromSavegame = true

                    if numObjects > 0 then
                        local customObjects = {}
                        local commonFilename = xmlFile:getValue(areaKey .. "#commonFilename")

                        if commonFilename ~= nil then
                            commonFilename = NetworkUtil.convertFromNetworkFilename(commonFilename)

                            if fileExists(commonFilename) then
                                defaultAttributes.xmlFilename = commonFilename
                            end
                        end

                        xmlFile:iterate(areaKey .. ".object", function (_, objectKey)
                            local saveIndex = xmlFile:getValue(objectKey .. "#index")

                            if saveIndex ~= nil and saveIndex <= numObjects then
                                local xmlFilename = xmlFile:getValue(objectKey .. "#filename")

                                if xmlFilename ~= nil then
                                    xmlFilename = NetworkUtil.convertFromNetworkFilename(xmlFilename)

                                    if not fileExists(xmlFilename) then
                                        xmlFilename = defaultAttributes.xmlFilename
                                    end
                                else
                                    xmlFilename = defaultAttributes.xmlFilename
                                end

                                local attributes = {
                                    farmId = farmId,
                                    fillType = fillTypeIndex,
                                    xmlFilename = xmlFilename,
                                    fillLevel = xmlFile:getValue(objectKey .. "#fillLevel", defaultAttributes.fillLevel),
                                    capacity = xmlFile:getValue(objectKey .. "#capacity", defaultAttributes.capacity),
                                    wrappingState = xmlFile:getValue(objectKey .. "#wrappingState", defaultAttributes.wrappingState),
                                    wrappingColor = xmlFile:getValue(objectKey .. "#wrappingColor", defaultAttributes.wrappingColor, true),
                                    baleValueScale = xmlFile:getValue(objectKey .. "#valueScale", defaultAttributes.baleValueScale),
                                    isMissionBale = xmlFile:getValue(objectKey .. "#isMissionBale", false)
                                }

                                local fillTypeName = xmlFile:getValue(objectKey .. "#fillType")

                                if fillTypeName ~= nil then
                                    attributes.fillType = g_fillTypeManager:getFillTypeIndexByName(fillTypeName) or fillTypeIndex
                                end

                                attributes.isFermenting = xmlFile:getValue(objectKey .. "#isFermenting", false)
                                attributes.fermentationTime = xmlFile:getValue(objectKey .. "#fermentationTime", 0)

                                if attributes.isFermenting then
                                    attributes.fermentingMaxTime = xmlFile:getValue(objectKey .. "#fermentationMaxTime", ObjectStorageManager.FERMENTING_FACTOR)
                                    attributes.maxFermentingTime = attributes.fermentingMaxTime * economicDifficulty
                                    attributes.fermentingPercentage = attributes.fermentationTime / attributes.maxFermentingTime
                                end

                                local wrapDiffuse = xmlFile:getValue(objectKey .. ".textures#wrapDiffuse")

                                if wrapDiffuse ~= nil then
                                    attributes.wrapDiffuse = NetworkUtil.convertFromNetworkFilename(wrapDiffuse)
                                end

                                local wrapNormal = xmlFile:getValue(objectKey .. ".textures#wrapNormal")

                                if wrapNormal ~= nil then
                                    attributes.wrapNormal = NetworkUtil.convertFromNetworkFilename(wrapNormal)
                                end

                                customObjects[saveIndex] = attributes
                            end
                        end)

                        storageArea.numObjects = numObjects

                        for i = 1, numObjects do
                            storageArea.objects[i] = customObjects[i] or defaultAttributes
                        end

                        self:onStorageLevelUpdateFinished(storageArea)
                    end
                end
            end
        end
    end
end

function BaleObjectStorage:saveToXMLFile(xmlFile, key, usedModNames)
    BaleObjectStorage:superClass().saveToXMLFile(self, xmlFile, key, usedModNames)

    local farmId = self.owningPlaceable:getOwnerFarmId()

    for areaIndex, storageArea in ipairs (self.indexedStorageAreas) do
        local areaKey = string.format("%s.area(%d)", key, areaIndex - 1)

        local numObjects = #storageArea.objects
        local fillTypeName = g_fillTypeManager:getFillTypeNameByIndex(storageArea.fillTypeIndex)

        xmlFile:setValue(areaKey .. "#numObjects", numObjects)
        xmlFile:setValue(areaKey .. "#fillType", fillTypeName or "UNKNOWN")

        local defaultAttributes = g_objectStorageManager:getBaleAttributesByObjectType(storageArea.objectType, storageArea.fillTypeIndex, farmId)

        if defaultAttributes ~= nil then
            local defaultFilename = defaultAttributes.xmlFilename
            local commonFilename = ObjectStorage.getCommonFilename(storageArea.objects, defaultFilename)

            xmlFile:setValue(areaKey .. "#defaultFilename", HTMLUtil.encodeToHTML(NetworkUtil.convertToNetworkFilename(defaultFilename)))

            if commonFilename ~= defaultFilename then
                xmlFile:setValue(areaKey .. "#commonFilename", HTMLUtil.encodeToHTML(NetworkUtil.convertToNetworkFilename(commonFilename)))
            end

            if numObjects > 0 then
                local objectIndex = 0

                for i, attributes in ipairs (storageArea.objects) do
                    local baleKey = string.format("%s.object(%d)", areaKey, objectIndex)
                    local saveIndex = false

                    if attributes.xmlFilename ~= commonFilename then
                        xmlFile:setValue(baleKey .. "#filename", HTMLUtil.encodeToHTML(NetworkUtil.convertToNetworkFilename(attributes.xmlFilename)))
                        saveIndex = true
                    end

                    if attributes.baleValueScale ~= 1 then
                        xmlFile:setValue(baleKey .. "#valueScale", attributes.baleValueScale)
                        saveIndex = true
                    end

                    local fillLevel = attributes.fillLevel
                    local capacity = attributes.capacity

                    if fillLevel < capacity or fillLevel ~= defaultAttributes.fillLevel then
                        xmlFile:setValue(baleKey .. "#fillLevel", fillLevel)
                        saveIndex = true
                    end

                    if capacity ~= defaultAttributes.capacity then
                        xmlFile:setValue(baleKey .. "#capacity", capacity)
                        saveIndex = true
                    end

                    local objectFillTypeName = g_fillTypeManager:getFillTypeNameByIndex(attributes.fillType)

                    if objectFillTypeName ~= nil and objectFillTypeName ~= fillTypeName then
                        xmlFile:setValue(baleKey .. "#fillType", objectFillTypeName)
                        saveIndex = true
                    end

                    if attributes.isMissionBale then
                        xmlFile:setValue(baleKey .. "#isMissionBale", attributes.isMissionBale)
                        saveIndex = true
                    end

                    local wrappingState = attributes.wrappingState or 0

                    if wrappingState > 0 then
                        local wrappingColor = attributes.wrappingColor or defaultAttributes.wrappingColor

                        xmlFile:setValue(baleKey .. "#wrappingState", wrappingState)
                        xmlFile:setValue(baleKey .. "#wrappingColor", wrappingColor[1], wrappingColor[2], wrappingColor[3], wrappingColor[4])

                        saveIndex = true
                    end

                    if attributes.isFermenting then
                        local fermentingMaxTime = attributes.fermentingMaxTime or ObjectStorageManager.FERMENTING_FACTOR

                        xmlFile:setValue(baleKey .. "#isFermenting", true)
                        xmlFile:setValue(baleKey .. "#fermentationTime", attributes.fermentationTime or 0)

                        if fermentingMaxTime ~= ObjectStorageManager.FERMENTING_FACTOR then
                            xmlFile:setValue(baleKey .. "#fermentationMaxTime", fermentingMaxTime)
                        end

                        saveIndex = true
                    end

                    if attributes.wrapDiffuse ~= nil then
                        xmlFile:setValue(baleKey .. ".textures#wrapDiffuse", HTMLUtil.encodeToHTML(NetworkUtil.convertToNetworkFilename(attributes.wrapDiffuse)))
                        saveIndex = true
                    end

                    if attributes.wrapNormal ~= nil then
                        xmlFile:setValue(baleKey .. ".textures#wrapNormal", HTMLUtil.encodeToHTML(NetworkUtil.convertToNetworkFilename(attributes.wrapNormal)))
                        saveIndex = true
                    end

                    if saveIndex then
                        xmlFile:setValue(baleKey .. "#index", i)
                        objectIndex = objectIndex + 1
                    end
                end
            end
        else

        end
    end
end

function BaleObjectStorage:onStorageObjectTypeChanged(storageArea)
    storageArea.supportsFermenting = g_objectStorageManager:getIsFermentedFillType(storageArea.fillTypeIndex)

    local storageAreas = {}

    for _, area in ipairs (self.indexedStorageAreas) do
        if area.fillTypeIndex ~= FillType.UNKNOWN and area.objectType ~= nil then
            local fillType = area.fillTypeIndex
            local typeIndex = area.objectType.index

            if storageAreas[fillType] == nil then
                storageAreas[fillType] = {}
            end

            if storageAreas[fillType][typeIndex] == nil then
                storageAreas[fillType][typeIndex] = {}
            end

            table.addElement(storageAreas[fillType][typeIndex], area)
        end
    end

    self.storageAreasByFillType = storageAreas

    BaleObjectStorage:superClass().onStorageObjectTypeChanged(self, storageArea)
end

function BaleObjectStorage:addToStorage(storageArea, attributes)
    if storageArea == nil or attributes == nil or storageArea.numObjects >= storageArea.maxObjects then
        return false
    end

    local numObjects = #storageArea.objects + 1

    storageArea.objects[numObjects] = attributes
    storageArea.numObjects = numObjects

    if self.isServer then
        if attributes.isFermenting then
            attributes.maxFermentingTime = attributes.fermentingMaxTime * g_currentMission.missionInfo.economicDifficulty
            g_objectStorageManager:addFermentingStorage(self)
        end

        g_server:broadcastEvent(ObjectStorageAddBaleObjectEvent.new(self, storageArea.index, attributes))
    end

    self:raiseStorageUpdate(storageArea)

    return true
end

function BaleObjectStorage:onObjectStorageAdded()
    for _, storageArea in ipairs (self.indexedStorageAreas) do
        if self:getHasFermentingBales(storageArea) then
            g_objectStorageManager:addFermentingStorage(self)

            break
        end
    end
end

function BaleObjectStorage:updateFermentation(dt, effectiveTimeScale)
    local requiresUpdate, syncFermentation = false, false

    for _, storageArea in ipairs (self.indexedStorageAreas) do
        if storageArea.supportsFermenting then
            for i = #storageArea.objects, 1, -1 do
                local attributes = storageArea.objects[i]

                if attributes.isFermenting then
                    attributes.fermentationTime = attributes.fermentationTime + dt * effectiveTimeScale

                    if attributes.fermentationTime >= attributes.maxFermentingTime then
                        attributes.isFermenting = false
                        attributes.fermentingPercentage = 1
                        attributes.fillType = storageArea.fillTypeIndex

                        syncFermentation = true
                    else
                        local fermentingPercentage = attributes.fermentationTime / attributes.maxFermentingTime

                        if math.floor(fermentingPercentage * 100) ~= math.floor(attributes.fermentingPercentage * 100) then
                            attributes.fermentingPercentage = fermentingPercentage

                            syncFermentation = true
                        end
                    end

                    requiresUpdate = true
                end
            end
        end
    end

    if syncFermentation then
        self:raiseDirtyFlags(self.fermentingDirtyFlag)
    end

    return requiresUpdate
end

function BaleObjectStorage:getFermentingBales(storageArea)
    if storageArea ~= nil and storageArea.supportsFermenting then
        local fermentingBales = {}

        for i = 1, #storageArea.objects do
            if storageArea.objects[i].isFermenting then
                table.insert(fermentingBales, {
                    fermentingPercentage = storageArea.objects[i].fermentingPercentage or 0,
                    index = i
                })
            end
        end

        return fermentingBales
    end

    return nil
end

function BaleObjectStorage:getHasFermentingBales(storageArea)
    if storageArea.supportsFermenting then
        for i = #storageArea.objects, 1, -1 do
            if storageArea.objects[i].isFermenting then
                return true
            end
        end
    end

    return false
end

function BaleObjectStorage:getBaleObjectTypeIndex(bale)
    for index, baleType in ipairs(self.availableObjectTypes) do
        if baleType.isRoundbale == bale.isRoundbale then
            if baleType.isRoundbale then
                if baleType.width == MathUtil.round(bale.width, 2) and baleType.diameter == MathUtil.round(bale.diameter, 2) then
                    return index
                end
            else
                if baleType.width == MathUtil.round(bale.width, 2) and baleType.height == MathUtil.round(bale.height, 2) and baleType.length == MathUtil.round(bale.length, 2) then
                    return index
                end
            end
        end
    end

    return 0
end

function BaleObjectStorage:updateInfo(infoTable)
    local numDisplayed = 0

    for i = 1, #self.indexedStorageAreas do
        local storageArea = self.indexedStorageAreas[i]

        if storageArea.fillTypeIndex ~= FillType.UNKNOWN then
            local fillTypeTitle = g_fillTypeManager:getFillTypeTitleByIndex(storageArea.fillTypeIndex)
            local numFermenting = 0
            local size = 0

            if storageArea.objectType ~= nil then
                size = (storageArea.objectType.isRoundbale and storageArea.objectType.diameter or storageArea.objectType.length) or 0
            end

            table.insert(infoTable, {
                title = string.format("%s (%d cm)", fillTypeTitle, size * 100),
                text = string.format("%d / %d", storageArea.numObjects, storageArea.maxObjects)
            })

            if storageArea.supportsFermenting then
                for i = #storageArea.objects, 1, -1 do
                    if storageArea.objects[i].isFermenting then
                        numFermenting = numFermenting + 1
                    end
                end
            end

            if numFermenting > 0 then
                table.insert(infoTable, {
                    accentuate = true,
                    title = self.infoTableFermentingText,
                    text = string.format("%d %s", numFermenting, self.infoTableBaleText)
                })
            end

            numDisplayed = numDisplayed + 1
        end
    end

    if numDisplayed == 0 then
        BaleObjectStorage:superClass().updateInfo(self, infoTable)
    end

    if not self.inputTriggerState then
        table.insert(infoTable, self.infoInputTriggerState)
    end
end

function BaleObjectStorage:inputTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
    if self.isServer and self.processBalesInTrigger then
        local object = g_currentMission:getNodeObject(otherId)

        if object ~= nil and object:isa(Bale) then
            if onEnter then
                if self.inputTriggerState then
                    if g_currentMission.accessHandler:canFarmAccessOtherId(self:getOwnerFarmId(), object:getOwnerFarmId()) then
                        self:raiseActive()
                        table.addElement(self.balesInTrigger, object)
                    else
                        self:raiseWarningMessage(ObjectStorageWarningMessageEvent.INVALID_FARM_MESSAGE, FillType.UNKNOWN, true)
                    end
                end
            else
                for index, bale in ipairs(self.balesInTrigger) do
                    if bale == object then
                        table.remove(self.balesInTrigger, index)

                        break
                    end
                end
            end
        end
    end
end
