--[[
Copyright (C) GtX (Andy), 2019

Author: GtX | Andy
Date: 25.08.2019
Revision: FS22-02

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

PalletObjectStorage = {}

PalletObjectStorage.MOD_NAME = g_currentModName
PalletObjectStorage.MOD_DIR = g_currentModDirectory

PalletObjectStorage.CONFIGURATION_KEYS = {
    fillUnit = "#fillUnitConfig",
    fillVolume = "#fillVolumeConfig",
    design = "#designConfig",
    treeSaplingType = "#treeSaplingConfig",
}

local PalletObjectStorage_mt = Class(PalletObjectStorage, ObjectStorage)

local function getIsValidPalletType(typeName)
    -- Backup in case of base game changes. All pallets should use the specialisation 'pallet' as this adds the BOOLEAN 'self.isPallet' allowing for easy identification
    return typeName == "pallet" or typeName == "bigBag" or typeName == "treeSaplingPallet"
end

InitObjectClass(PalletObjectStorage, "PalletObjectStorage")

function PalletObjectStorage.registerXMLPaths(schema, basePath)
end

function PalletObjectStorage.registerSavegameXMLPaths(schema, basePath)
    schema:register(XMLValueType.INT, basePath .. ".area(?)#defaultCapacity", "Default capacity")
    schema:register(XMLValueType.INT, basePath .. ".area(?).object(?)#fillUnitConfig", "Fill Unit configuration id when > 1")
    schema:register(XMLValueType.INT, basePath .. ".area(?).object(?)#fillVolumeConfig", "Fill Volume configuration id when > 1")
    schema:register(XMLValueType.INT, basePath .. ".area(?).object(?)#designConfig", "Design configuration id when > 1")
    schema:register(XMLValueType.INT, basePath .. ".area(?).object(?)#treeSaplingConfig", "Tree sapling configuration id when > 1")
end

function PalletObjectStorage.new(isServer, isClient, baseDirectory, customEnvironment, customMt)
    local self = ObjectStorage.new(isServer, isClient, baseDirectory, customEnvironment, customMt or PalletObjectStorage_mt)

    self.name = string.format("%s %s", g_i18n:getText("infohud_pallet"), g_i18n:getText("statistic_storage"))

    self.baleStorage = false
    self.palletStorage = true

    return self
end

function PalletObjectStorage:load(xmlFile, key, components, i3dMappings)
    if not PalletObjectStorage:superClass().load(self, xmlFile, key, components, i3dMappings) then
        return false
    end

    local availableObjectTypes = g_objectStorageManager:getAvailablePalletObjectTypes()
    local availableFillTypes = g_objectStorageManager:getAvailablePalletFillTypes()

    if (availableObjectTypes == nil or #availableObjectTypes == 0) or (availableFillTypes == nil or table.size(availableFillTypes) == 0) then
        Logging.xmlError(xmlFile, "No valid object types defined!")

        return false
    end

    local acceptedFillTypes = nil
    local fillTypeNames = xmlFile:getValue(key .. ".storageAreas#fillTypes")

    if fillTypeNames ~= nil then
        fillTypeNames = string.split(fillTypeNames, " ")

        if #fillTypeNames > 0 then
            acceptedFillTypes = {}

            for _, fillTypeName in pairs(fillTypeNames) do
                local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(fillTypeName)

                if fillTypeIndex ~= nil and availableFillTypes[fillTypeIndex] then
                    acceptedFillTypes[fillTypeIndex] = true
                end
            end
        end
    end

    if acceptedFillTypes == nil then
        acceptedFillTypes = {}

        for fillTypeIndex, _ in pairs(availableFillTypes) do
            acceptedFillTypes[fillTypeIndex] = true
        end
    end

    local sortedAcceptedTypes = g_objectStorageManager:getSortedAcceptedPalletTypes(acceptedFillTypes)

    if #sortedAcceptedTypes == 0 then
        Logging.xmlError(xmlFile, "Object storage (%s) does not support any available fill types!", self.owningPlaceable.customEnvironment)

        return false
    end

    self.availableObjectTypes = availableObjectTypes

    self.acceptedFillTypes = acceptedFillTypes
    self.sortedAcceptedTypes = sortedAcceptedTypes

    self:loadSharedVisibilityNodes(xmlFile, key, acceptedFillTypes, true)
    self:loadStorageAreas(xmlFile, key, components, i3dMappings, acceptedFillTypes, sortedAcceptedTypes)

    return true
end

function PalletObjectStorage:readStream(streamId, connection)
    PalletObjectStorage:superClass().readStream(self, streamId, connection)

    if connection:getIsServer() then
        for _, storageArea in ipairs (self.indexedStorageAreas) do
            storageArea.numObjects = streamReadUIntN(streamId, ObjectStorage.OBJECTS_SEND_NUM_BITS)
            storageArea.fillTypeIndex = streamReadUIntN(streamId, FillTypeManager.SEND_NUM_BITS)

            local objectTypeIndex = streamReadUIntN(streamId, ObjectStorage.TYPE_SEND_NUM_BITS)
            storageArea.objectType = self:getObjectType(objectTypeIndex, storageArea.fillTypeIndex)

            self:setVisibilityNodes(storageArea, self.onStorageObjectTypeChanged, self)

            if storageArea.numObjects > 0 then
                for i = 1, storageArea.numObjects do
                    local xmlFilename = NetworkUtil.convertFromNetworkFilename(streamReadString(streamId))
                    local fillLevel = streamReadFloat32(streamId)
                    local capacity = streamReadFloat32(streamId)

                    storageArea.objects[i] = {
                        xmlFilename = xmlFilename,
                        fillLevel = fillLevel,
                        capacity = capacity
                    }
                end
            end

            self:raiseStorageUpdate(storageArea)
        end
    end
end

function PalletObjectStorage:writeStream(streamId, connection)
    PalletObjectStorage:superClass().writeStream(self, streamId, connection)

    if not connection:getIsServer() then
        for _, storageArea in ipairs (self.indexedStorageAreas) do
            storageArea.numObjects = #storageArea.objects

            streamWriteUIntN(streamId, storageArea.numObjects, ObjectStorage.OBJECTS_SEND_NUM_BITS)
            streamWriteUIntN(streamId, storageArea.fillTypeIndex, FillTypeManager.SEND_NUM_BITS)

            local objectTypeIndex = storageArea.objectType and storageArea.objectType.index or 0
            streamWriteUIntN(streamId, objectTypeIndex, ObjectStorage.TYPE_SEND_NUM_BITS)

            if storageArea.numObjects > 0 then
                for i = 1, storageArea.numObjects do
                    local attributes = storageArea.objects[i]
                    local fillLevel = attributes.fillLevel or 0

                    streamWriteString(streamId, NetworkUtil.convertToNetworkFilename(attributes.xmlFilename or ""))
                    streamWriteFloat32(streamId, fillLevel)
                    streamWriteFloat32(streamId, attributes.capacity or fillLevel)
                end
            end
        end
    end
end

function PalletObjectStorage:loadFromXMLFile(xmlFile, key)
    PalletObjectStorage:superClass().loadFromXMLFile(self, xmlFile, key)

    local palletFilenames = {}

    local function getPalletIsValid(filename)
        if palletFilenames[filename] == nil then
            if g_storeManager:getItemByXMLFilename(filename) == nil then
                palletFilenames[filename] = false
            else
                palletFilenames[filename] = true
            end
        end

        return palletFilenames[filename]
    end

    for areaIndex, storageArea in ipairs (self.indexedStorageAreas) do
        local areaKey = string.format("%s.area(%d)", key, areaIndex - 1)

        local defaultFilename = xmlFile:getValue(areaKey .. "#defaultFilename")
        local fillTypeIndex = g_fillTypeManager:getFillTypeIndexByName(xmlFile:getValue(areaKey .. "#fillType"), "INVALID")

        if defaultFilename ~= nil and self.acceptedFillTypes[fillTypeIndex] then
            local defaultAttributes, pallet = g_objectStorageManager:getPalletAttributesByFilename(NetworkUtil.convertFromNetworkFilename(defaultFilename), fillTypeIndex)

            if defaultAttributes == nil then
                defaultAttributes, pallet = g_objectStorageManager:getPalletAttributesByFillType(fillTypeIndex)

                if pallet ~= nil then
                    defaultAttributes.capacity = xmlFile:getValue(areaKey .. "#defaultCapacity", defaultAttributes.capacity)
                    defaultAttributes.fillLevel = defaultAttributes.capacity
                end
            end

            if defaultAttributes ~= nil then
                if self:setStorageObjectType(areaIndex, pallet.index, fillTypeIndex, true) then
                    local numObjects = MathUtil.clamp(xmlFile:getValue(areaKey .. "#numObjects", storageArea.numObjects), 0, storageArea.maxObjects)

                    storageArea.loadedFromSavegame = true

                    if numObjects > 0 then
                        local customObjects = {}
                        local commonFilename = xmlFile:getValue(areaKey .. "#commonFilename")

                        if commonFilename ~= nil then
                            commonFilename = NetworkUtil.convertFromNetworkFilename(commonFilename)

                            if getPalletIsValid(commonFilename) then
                                defaultAttributes.xmlFilename = commonFilename
                            end
                        end

                        xmlFile:iterate(areaKey .. ".object", function (_, objectKey)
                            local saveIndex = xmlFile:getValue(objectKey .. "#index")

                            if saveIndex ~= nil and saveIndex <= numObjects then
                                local xmlFilename = xmlFile:getValue(objectKey .. "#filename")

                                if xmlFilename ~= nil then
                                    xmlFilename = NetworkUtil.convertFromNetworkFilename(xmlFilename)

                                    if not getPalletIsValid(xmlFilename) then
                                        xmlFilename = defaultAttributes.xmlFilename
                                    end
                                else
                                    xmlFilename = defaultAttributes.xmlFilename
                                end

                                local attributes = {
                                    fillType = fillTypeIndex,
                                    xmlFilename = xmlFilename,
                                    capacity = xmlFile:getValue(objectKey .. "#capacity", defaultAttributes.capacity),
                                    fillLevel = xmlFile:getValue(objectKey .. "#fillLevel", defaultAttributes.fillLevel)
                                }

                                for configName, configKey in pairs (PalletObjectStorage.CONFIGURATION_KEYS) do
                                    local configId = xmlFile:getValue(objectKey .. configKey)

                                    if configId ~= nil then
                                        if attributes.configurations == nil then
                                            attributes.configurations = {}
                                        end

                                        attributes.configurations[configName] = configId
                                    end
                                end

                                customObjects[saveIndex] = attributes
                            end
                        end)

                        for i = 1, numObjects do
                            self:addToStorage(storageArea, customObjects[i] or defaultAttributes)
                        end

                        self:onStorageLevelUpdateFinished(storageArea)
                    end
                end
            end
        end
    end
end

function PalletObjectStorage:saveToXMLFile(xmlFile, key, usedModNames)
    PalletObjectStorage:superClass().saveToXMLFile(self, xmlFile, key, usedModNames)

    for areaIndex, storageArea in ipairs (self.indexedStorageAreas) do
        local areaKey = string.format("%s.area(%d)", key, areaIndex - 1)

        local numObjects = #storageArea.objects
        local fillTypeName = g_fillTypeManager:getFillTypeNameByIndex(storageArea.fillTypeIndex)

        xmlFile:setValue(areaKey .. "#numObjects", numObjects)
        xmlFile:setValue(areaKey .. "#fillType", fillTypeName or "UNKNOWN")

        local defaultAttributes = g_objectStorageManager:getPalletAttributesByFillType(storageArea.fillTypeIndex)

        if defaultAttributes ~= nil then
            local defaultFilename = defaultAttributes.xmlFilename
            local commonFilename = ObjectStorage.getCommonFilename(storageArea.objects, defaultFilename)

            xmlFile:setValue(areaKey .. "#defaultFilename", HTMLUtil.encodeToHTML(NetworkUtil.convertToNetworkFilename(defaultFilename)))

            if commonFilename ~= defaultFilename then
                xmlFile:setValue(areaKey .. "#commonFilename", HTMLUtil.encodeToHTML(NetworkUtil.convertToNetworkFilename(commonFilename)))
            end

            xmlFile:setValue(areaKey .. "#defaultCapacity", defaultAttributes.capacity)

            if numObjects > 0 then
                local objectIndex = 0

                for i, attributes in ipairs (storageArea.objects) do
                    local palletKey = string.format("%s.object(%d)", areaKey, objectIndex)
                    local saveIndex = false

                    if attributes.xmlFilename ~= commonFilename then
                        xmlFile:setValue(palletKey .. "#filename", HTMLUtil.encodeToHTML(NetworkUtil.convertToNetworkFilename(attributes.xmlFilename)))
                        saveIndex = true
                    end

                    local fillLevel = attributes.fillLevel
                    local capacity = attributes.capacity

                    if fillLevel < capacity or fillLevel ~= defaultAttributes.fillLevel then
                        xmlFile:setValue(palletKey .. "#fillLevel", fillLevel)
                        saveIndex = true
                    end

                    if capacity ~= defaultAttributes.capacity then
                        xmlFile:setValue(palletKey .. "#capacity", capacity)
                        saveIndex = true
                    end

                    if attributes.configurations ~= nil then
                        for configName, configId in pairs(attributes.configurations) do
                            local configKey = PalletObjectStorage.CONFIGURATION_KEYS[configName]

                            if configKey ~= nil and configId > 1 then
                                xmlFile:setValue(palletKey .. configKey, configId)
                                saveIndex = true
                            end
                        end
                    end

                    if saveIndex then
                        xmlFile:setValue(palletKey .. "#index", i)
                        objectIndex = objectIndex + 1
                    end
                end
            end
        else

        end
    end
end

function PalletObjectStorage:onStorageObjectTypeChanged(storageArea)
    local storageAreas = {}

    for _, area in ipairs (self.indexedStorageAreas) do
        if area.fillTypeIndex ~= FillType.UNKNOWN and area.objectType ~= nil then
            local fillType = area.fillTypeIndex

            if storageAreas[fillType] == nil then
                storageAreas[fillType] = {}
            end

            table.addElement(storageAreas[fillType], area)
        end
    end

    self.storageAreasByFillType = storageAreas

    PalletObjectStorage:superClass().onStorageObjectTypeChanged(self, storageArea)
end

function PalletObjectStorage:updateInfo(infoTable)
    local numDisplayed = 0

    for i = 1, #self.indexedStorageAreas do
        local storageArea = self.indexedStorageAreas[i]

        if storageArea.fillTypeIndex ~= FillType.UNKNOWN then
            table.insert(infoTable, {
                title = g_fillTypeManager:getFillTypeTitleByIndex(storageArea.fillTypeIndex) or "",
                text = string.format("%d / %d", storageArea.numObjects, storageArea.maxObjects)
            })

            numDisplayed = numDisplayed + 1
        end
    end

    if numDisplayed == 0 then
        PalletObjectStorage:superClass().updateInfo(self, infoTable)
    end

    if not self.inputTriggerState then
        table.insert(infoTable, self.infoInputTriggerState)
    end
end

function PalletObjectStorage:getObjectType(index, fillTypeIndex)
    local availableObjectType = self.availableObjectTypes[index]

    if availableObjectType == nil and fillTypeIndex ~= nil then
        for _, objectType in ipairs (self.availableObjectTypes) do
            if objectType.fillTypeIndexs[fillTypeIndex] then
                return objectType
            end
        end
    end

    return availableObjectType
end

function PalletObjectStorage:getObjectIsValid(object)
    if object == nil or not object:isa(Vehicle) or object.getFillUnitExists == nil then
        return false
    end

    if object.isPallet or getIsValidPalletType(object.typeName) then
        return object:getFillUnitExists(1)
    end

    if object.spec_wheels == nil and object.spec_enterable == nil then
        for _, spec in pairs(object.specializations) do
            if spec == Pallet or spec == BigBag or spec == TreeSaplingPallet then
                return object:getFillUnitExists(1)
            end
        end
    end

    return false
end

function PalletObjectStorage:inputTriggerCallback(triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
    if self.isServer and self.inputTriggerState and onEnter then
        local object = g_currentMission:getNodeObject(otherId)

        if self:getObjectIsValid(object) then
            if not g_currentMission.accessHandler:canFarmAccessOtherId(self:getOwnerFarmId(), object:getOwnerFarmId()) then
                self:raiseWarningMessage(ObjectStorageWarningMessageEvent.INVALID_FARM_MESSAGE, FillType.UNKNOWN, true)

                return
            end

            local fillUnit = object:getFillUnitByIndex(1)

            if fillUnit ~= nil and fillUnit.fillLevel > 0.01 then
                local storageArea = self:getValidStorageArea(fillUnit.fillType, 1, true)

                if storageArea ~= nil then
                    local size = object.size

                    if self.spawnArea == nil or not self.spawnArea:getCanSpawnSize(size.width, size.height, size.length) then
                        self:raiseWarningMessage(ObjectStorageWarningMessageEvent.INVALID_SIZE_MESSAGE, fillUnit.fillType, true)

                        return
                    end

                    local attributes = {
                        xmlFilename = object.configFileName,
                        fillLevel = fillUnit.fillLevel,
                        capacity = fillUnit.capacity
                    }

                    if object.configurations ~= nil then
                        for configName, configId in pairs(object.configurations) do
                            if PalletObjectStorage.CONFIGURATION_KEYS[configName] ~= nil and configId > 1 then
                                if attributes.configurations == nil then
                                    attributes.configurations = {}
                                end

                                attributes.configurations[configName] = configId
                            end
                        end
                    end

                    if self:addToStorage(storageArea, attributes) then
                        g_currentMission:removeVehicle(object)
                    end
                end
            end
        end
    end
end
