--[[
Copyright (C) GtX (Andy), 2022

Author: GtX | Andy
Date: 02.01.2022
Revision: FS22-03

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

ObjectStoragePlaceable = {}

ObjectStoragePlaceable.MOD_NAME = g_currentModName or ""
ObjectStoragePlaceable.SPEC_NAME = string.format("%s.objectStorage", ObjectStoragePlaceable.MOD_NAME)
ObjectStoragePlaceable.SPEC = string.format("spec_%s", ObjectStoragePlaceable.SPEC_NAME)

local specEntryName = ObjectStoragePlaceable.SPEC

function ObjectStoragePlaceable.initSpecialization()
    g_storeManager:addSpecType("objectStorageModCapacity", "constructionListAttributeIconCapacity", ObjectStoragePlaceable.loadSpecValueCapacity, ObjectStoragePlaceable.getSpecValueCapacity, "placeable")
    g_storeManager:addSpecType("objectStorageModAreas", "constructionListAttributeIconAreas", ObjectStoragePlaceable.loadSpecValueAreas, ObjectStoragePlaceable.getSpecValueAreas, "placeable")
    g_storeManager:addSpecType("objectStorageModSizeRoundBale", "constructionListAttributeIconSizeRoundBale", ObjectStoragePlaceable.loadSpecValueBaleSizeRound, ObjectStoragePlaceable.getSpecValueBaleSizeRound, "placeable")
    g_storeManager:addSpecType("objectStorageModSizeSquareBale", "constructionListAttributeIconSizeSquareBale", ObjectStoragePlaceable.loadSpecValueBaleSizeSquare, ObjectStoragePlaceable.getSpecValueBaleSizeSquare, "placeable")
end

function ObjectStoragePlaceable.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(PlaceableInfoTrigger, specializations)
end

function ObjectStoragePlaceable.registerOverwrittenFunctions(placeableType)
    SpecializationUtil.registerOverwrittenFunction(placeableType, "setName", ObjectStoragePlaceable.setName)
    SpecializationUtil.registerOverwrittenFunction(placeableType, "setOwnerFarmId", ObjectStoragePlaceable.setOwnerFarmId)
    SpecializationUtil.registerOverwrittenFunction(placeableType, "updateInfo", ObjectStoragePlaceable.updateInfo)
    SpecializationUtil.registerOverwrittenFunction(placeableType, "canBuy", ObjectStoragePlaceable.canBuy)
    SpecializationUtil.registerOverwrittenFunction(placeableType, "canBeSold", ObjectStoragePlaceable.canBeSold)
end

function ObjectStoragePlaceable.registerEventListeners(placeableType)
    SpecializationUtil.registerEventListener(placeableType, "onPreLoad", ObjectStoragePlaceable)
    SpecializationUtil.registerEventListener(placeableType, "onLoad", ObjectStoragePlaceable)
    SpecializationUtil.registerEventListener(placeableType, "onDelete", ObjectStoragePlaceable)
    SpecializationUtil.registerEventListener(placeableType, "onFinalizePlacement", ObjectStoragePlaceable)
    SpecializationUtil.registerEventListener(placeableType, "onWriteStream", ObjectStoragePlaceable)
    SpecializationUtil.registerEventListener(placeableType, "onReadStream", ObjectStoragePlaceable)
end

function ObjectStoragePlaceable.registerXMLPaths(schema, basePath)
    schema:setXMLSpecializationType("ObjectStorageMod")

    schema:register(XMLValueType.INT, "placeable.storeData.specs.objectStorageAreas", "Number of storage areas")
    schema:register(XMLValueType.STRING, "placeable.storeData.specs.objectStorageCapacity", "Capacity or capacity range. Example: 50 - 142")

    schema:register(XMLValueType.BOOL, basePath .. ".objectStorage#baleStorage", "Storage building is for bales", true)

    ObjectStorage.registerXMLPaths(schema, basePath .. ".objectStorage")

    schema:setXMLSpecializationType()
end

function ObjectStoragePlaceable.registerSavegameXMLPaths(schema, basePath)
    schema:setXMLSpecializationType("ObjectStorageMod")

    ObjectStorage.registerSavegameXMLPaths(schema, basePath)

    schema:setXMLSpecializationType()
end

function ObjectStoragePlaceable:onPreLoad(savegame)
    self.spec_objectStorageMod = self[specEntryName] -- Changed after game update 1.9.0.0 as new base game spec uses the same name :-(
end

function ObjectStoragePlaceable:onLoad(savegame)
    local spec = self[specEntryName]

    if g_objectStorageManager ~= nil then
        local objectStorageClass = PalletObjectStorage

        spec.baleStorage = self.xmlFile:getValue("placeable.objectStorage#baleStorage", true)

        if spec.baleStorage then
            objectStorageClass = BaleObjectStorage
        end

        if objectStorageClass ~= nil then
            local objectStorage = objectStorageClass.new(self.isServer, self.isClient, self.baseDirectory, self.customEnvironment)

            objectStorage.owningPlaceable = self

            if objectStorage:load(self.xmlFile, "placeable.objectStorage", self.components, self.i3dMappings) then
                spec.objectStorage = objectStorage
            else
                objectStorage:delete()
            end
        end
    end

    if spec.objectStorage == nil then
        SpecializationUtil.removeEventListener(self, "onDelete", ObjectStoragePlaceable)
        SpecializationUtil.removeEventListener(self, "onFinalizePlacement", ObjectStoragePlaceable)
        SpecializationUtil.removeEventListener(self, "onWriteStream", ObjectStoragePlaceable)
        SpecializationUtil.removeEventListener(self, "onReadStream", ObjectStoragePlaceable)
    end
end

function ObjectStoragePlaceable:onDelete()
    local spec = self[specEntryName]

    if spec.objectStorage ~= nil then
        spec.objectStorage:delete()
        spec.objectStorage = nil
    end
end

function ObjectStoragePlaceable:onFinalizePlacement()
    local spec = self[specEntryName]

    if spec.objectStorage ~= nil then
        spec.objectStorage:finalizePlacement()
        spec.objectStorage:register(true)
        spec.objectStorage:setOwnerFarmId(self:getOwnerFarmId())
    end
end

function ObjectStoragePlaceable:onReadStream(streamId, connection)
    local spec = self[specEntryName]

    local objectStorageId = NetworkUtil.readNodeObjectId(streamId)
    spec.objectStorage:readStream(streamId, connection)
    g_client:finishRegisterObject(spec.objectStorage, objectStorageId)
end

function ObjectStoragePlaceable:onWriteStream(streamId, connection)
    local spec = self[specEntryName]

    NetworkUtil.writeNodeObjectId(streamId, NetworkUtil.getObjectId(spec.objectStorage))
    spec.objectStorage:writeStream(streamId, connection)
    g_server:registerObjectInStream(connection, spec.objectStorage)
end

function ObjectStoragePlaceable:loadFromXMLFile(xmlFile, key)
    local spec = self[specEntryName]

    if spec.objectStorage ~= nil then
        spec.objectStorage:loadFromXMLFile(xmlFile, key)
    end
end

function ObjectStoragePlaceable:saveToXMLFile(xmlFile, key, usedModNames)
    local spec = self[specEntryName]

    if spec.objectStorage ~= nil then
        spec.objectStorage:saveToXMLFile(xmlFile, key, usedModNames)
    end
end

function ObjectStoragePlaceable:setName(superFunc, name, noEventSend)
    if superFunc(self, name, noEventSend) then
        local spec = self[specEntryName]

        if spec.objectStorage ~= nil then
            spec.objectStorage:onSetName(name)
        end

        return true
    end

    return false
end

function ObjectStoragePlaceable:setOwnerFarmId(superFunc, farmId, noEventSend)
    local spec = self[specEntryName]

    superFunc(self, farmId, noEventSend)

    if spec.objectStorage ~= nil then
        spec.objectStorage:setOwnerFarmId(farmId, noEventSend)
    end
end

function ObjectStoragePlaceable:updateInfo(superFunc, infoTable)
    local spec = self[specEntryName]

    superFunc(self, infoTable)

    if spec.objectStorage ~= nil then
        spec.objectStorage:updateInfo(infoTable)
    end
end

function ObjectStoragePlaceable:canBuy(superFunc)
    local spec = self[specEntryName]

    if spec.objectStorage == nil then
        return false, g_i18n:getText("warning_placeable_error_cannotBeBought")
    end

    if g_objectStorageManager ~= nil and not g_objectStorageManager:getCanAddStorage() then
        local limit = g_objectStorageManager:getMaxStorages()

        return false, string.format("%s  Max. %d", g_i18n:getText("warning_tooManyPlaceables"), limit)
    end

    return superFunc(self)
end

function ObjectStoragePlaceable:canBeSold(superFunc)
    local spec = self[specEntryName]

    if spec.objectStorage ~= nil then
        for _, storageArea in ipairs (spec.objectStorage.indexedStorageAreas) do
            if #storageArea.objects > 0 then
                local l10n = spec.baleStorage and "warning_objectStorage_demolishPlaceableBales" or "warning_objectStorage_demolishPlaceablePallets"

                return false, g_i18n:getText(l10n, ObjectStoragePlaceable.MOD_NAME)
            end
        end
    end

    return superFunc(self)
end

function ObjectStoragePlaceable.loadSpecValueAreas(xmlFile, customEnvironment, baseDir)
    return xmlFile:getValue("placeable.storeData.specs.objectStorageAreas")
end

function ObjectStoragePlaceable.getSpecValueAreas(storeItem, realItem)
    local numAreas = storeItem.specs.objectStorageModAreas

    if numAreas == nil or numAreas <= 0 then
        return nil
    end

    return tostring(numAreas)
end

function ObjectStoragePlaceable.loadSpecValueCapacity(xmlFile, customEnvironment, baseDir)
    return xmlFile:getValue("placeable.storeData.specs.objectStorageCapacity")
end

function ObjectStoragePlaceable.getSpecValueCapacity(storeItem, realItem)
    if storeItem.specs.objectStorageModCapacity == nil then
        return nil
    end

    return string.format("%s %s", storeItem.specs.objectStorageModCapacity, g_i18n:getText("unit_pieces"))
end

function ObjectStoragePlaceable.loadSpecValueBaleSizeRound(xmlFile, customEnvironment, baseDir)
    if xmlFile:getValue("placeable.objectStorage#baleStorage", false) then
        local minDiameter = MathUtil.round(xmlFile:getValue("placeable.objectStorage.storageAreas#minBaleDiameter", 0), 2)
        local maxDiameter = MathUtil.round(xmlFile:getValue("placeable.objectStorage.storageAreas#maxBaleDiameter", 0), 2)

        if minDiameter > 0 and maxDiameter > 0 then
            return {
                minDiameter = minDiameter,
                maxDiameter = maxDiameter
            }
        end
    end
end

function ObjectStoragePlaceable.loadSpecValueBaleSizeSquare(xmlFile, customEnvironment, baseDir)
    if xmlFile:getValue("placeable.objectStorage#baleStorage", false) then
        local minLength = MathUtil.round(xmlFile:getValue("placeable.objectStorage.storageAreas#minBaleLength", 0), 2)
        local maxLength = MathUtil.round(xmlFile:getValue("placeable.objectStorage.storageAreas#maxBaleLength", 0), 2)

        if minLength > 0 and maxLength > 0 then
            return {
                minLength = minLength,
                maxLength = maxLength
            }
        end
    end
end

function ObjectStoragePlaceable.getSpecValueBaleSize(storeItem, realItem, configurations, saleItem, returnValues, returnRange, roundBale)
    local baleSizeAttributes = roundBale and storeItem.specs.objectStorageModSizeRoundBale or storeItem.specs.objectStorageModSizeSquareBale

    if baleSizeAttributes ~= nil then
        local minValue = (roundBale and baleSizeAttributes.minDiameter or baleSizeAttributes.minLength) or 0
        local maxValue = (roundBale and baleSizeAttributes.maxDiameter or baleSizeAttributes.maxLength) or 0

        if minValue > 0 and maxValue > 0 then
            local unit = g_i18n:getText("unit_cmShort")

            if returnValues == nil or not returnValues then
                if minValue == maxValue then
                    return string.format("%d%s", minValue * 100, unit)
                end

                return string.format("%d%s-%d%s", minValue * 100, unit, maxValue * 100, unit)
            end

            if returnRange == true and minValue ~= maxValue then
                return minValue * 100, maxValue * 100, unit
            end

            return minValue * 100, unit
        end
    end

    if returnValues and returnRange then
        return 0, 0, ""
    end

    if returnValues then
        return 0, ""
    end

    return nil
end

function ObjectStoragePlaceable.getSpecValueBaleSizeRound(storeItem, realItem, configurations, saleItem, returnValues, returnRange)
    if storeItem.specs.objectStorageModSizeRoundBale ~= nil then
        return ObjectStoragePlaceable.getSpecValueBaleSize(storeItem, realItem, configurations, saleItem, returnValues, returnRange, true)
    end
end

function ObjectStoragePlaceable.getSpecValueBaleSizeSquare(storeItem, realItem, configurations, saleItem, returnValues, returnRange)
    if storeItem.specs.objectStorageModSizeSquareBale ~= nil then
        return ObjectStoragePlaceable.getSpecValueBaleSize(storeItem, realItem, configurations, saleItem, returnValues, returnRange, false)
    end
end

-- Unfortunately another fix for a bug created by latest game update where Giants have used the same key naming as me :-/
if PlaceableObjectStorage ~= nil and PlaceableObjectStorage.loadSpecValueCapacity ~= nil then
    local oldLoadSpecValueCapacity = PlaceableObjectStorage.loadSpecValueCapacity

    function PlaceableObjectStorage.loadSpecValueCapacity(xmlFile, customEnvironment, baseDir, ...)
        -- Base game 'Object Storage' uses the following XML key so return nil if it is missing to stop duplicate store capacity info.
        if not xmlFile:hasProperty("placeable.objectStorage.spawnAreas(0)") then
            return nil
        end

        return oldLoadSpecValueCapacity(xmlFile, customEnvironment, baseDir, ...)
    end
end
