--[[
Copyright (C) GtX (Andy), 2022

Author: GtX | Andy
Date: 14.07.2022
Revision: FS22-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

EnterablePassengerExtension = {}
EnterablePassengerExtension.xmlSchema = XMLSchema.new("enterablePassengerExtension")

local EnterablePassengerExtension_mt = Class(EnterablePassengerExtension)

g_xmlManager:addInitSchemaFunction(function ()
    local schema = EnterablePassengerExtension.xmlSchema

    schema:setXMLSpecializationType("EnterablePassengerExtension")
    schema:register(XMLValueType.STRING, "enterablePassengerExtension.vehicle(?)#xmlFilename", "Vehicle filename")
    schema:register(XMLValueType.STRING, "enterablePassengerExtension.vehicle(?)#prefix", "Type prefix. $moddir$ | $mapdir$ | $pdlcdir$")

    schema:register(XMLValueType.NODE_INDEX, "enterablePassengerExtension.vehicle(?).passengerSeat(?)#linkNode", "Optional link node for seat otherwise vehicle root node will be used.")
    schema:register(XMLValueType.INT, "enterablePassengerExtension.vehicle(?).passengerSeat(?)#outdoorCameraIndex", "Index of standard outdoor camera", 1)
    schema:register(XMLValueType.FLOAT, "enterablePassengerExtension.vehicle(?).passengerSeat(?)#nicknameOffset", "Nickname rendering offset", 1.5)
    schema:register(XMLValueType.VECTOR_TRANS, "enterablePassengerExtension.vehicle(?).passengerSeat(?)#exitNodePosition")
    schema:register(XMLValueType.BOOL, "enterablePassengerExtension.vehicle(?).passengerSeat(?)#disabledReverseDriving", "Disabled when reverse driving is active", false)

    local characterNodeKey = "enterablePassengerExtension.vehicle(?).passengerSeat(?).characterNode"

    schema:register(XMLValueType.VECTOR_TRANS, characterNodeKey .. "#position")
    schema:register(XMLValueType.VECTOR_ROT, characterNodeKey .. "#rotation")

    for _, ikName in pairs ({"rightFoot", "leftFoot", "rightArm", "leftArm"}) do
        schema:register(XMLValueType.VECTOR_TRANS, string.format("%s.%s#position", characterNodeKey, ikName))
        schema:register(XMLValueType.VECTOR_ROT, string.format("%s.%s#rotation", characterNodeKey, ikName))
        schema:register(XMLValueType.STRING, string.format("%s.%s#poseId", characterNodeKey, ikName))
    end

    schema:register(XMLValueType.FLOAT, characterNodeKey .. "#cameraMinDistance", "Min. distance until character is hidden", 0.5)
    schema:register(XMLValueType.VECTOR_ROT, characterNodeKey .. "#spineRotation", "Spine rotation")

    schema:register(XMLValueType.FLOAT, characterNodeKey .. "#maxUpdateDistance", "Max. distance to vehicle root to update ik chains of character", VehicleCharacter.DEFAULT_MAX_UPDATE_DISTANCE)
    schema:register(XMLValueType.FLOAT, characterNodeKey .. "#clipDistance", "Clip distance of character", VehicleCharacter.DEFAULT_CLIP_DISTANCE)

    local cameraNodeKey = "enterablePassengerExtension.vehicle(?).passengerSeat(?).camera"

    schema:register(XMLValueType.VECTOR_TRANS, cameraNodeKey .. "#cameraPosition")
    schema:register(XMLValueType.VECTOR_ROT, cameraNodeKey .. "#cameraRotation")
    schema:register(XMLValueType.BOOL, cameraNodeKey .. "#rotatable", "Camera is rotatable", true)
    schema:register(XMLValueType.BOOL, cameraNodeKey .. "#limit", "Has limits", true)
    schema:register(XMLValueType.FLOAT, cameraNodeKey .. "#rotMinX", "Min. X rotation")
    schema:register(XMLValueType.FLOAT, cameraNodeKey .. "#rotMaxX", "Max. X rotation")
    schema:register(XMLValueType.FLOAT, cameraNodeKey .. "#transMin", "Min. Z translation")
    schema:register(XMLValueType.FLOAT, cameraNodeKey .. "#transMax", "Max. Z translation")
    schema:register(XMLValueType.BOOL, cameraNodeKey .. "#isInside", "Is camera inside. Used for camera smoothing and fallback/default value for 'useOutdoorSounds'", false)
    schema:register(XMLValueType.BOOL, cameraNodeKey .. "#allowHeadTracking", "Allow head tracking", "isInside value")
    schema:register(XMLValueType.INT, cameraNodeKey .. "#shadowFocusBoxCameraIndex", "Index of camera with shadowFocusBox to use")
    schema:register(XMLValueType.NODE_INDEX, cameraNodeKey .. "#shadowFocusBox", "Shadow focus box")
    schema:register(XMLValueType.BOOL, cameraNodeKey .. "#useOutdoorSounds", "Use outdoor sounds", "false for 'isInside' cameras, otherwise true")
    schema:register(XMLValueType.VECTOR_ROT, cameraNodeKey .. "#rotation", "Camera rotation")
    schema:register(XMLValueType.VECTOR_TRANS, cameraNodeKey .. "#translation", "Camera translation")
    schema:register(XMLValueType.BOOL, cameraNodeKey .. "#useMirror", "Use mirrors", false)
    schema:register(XMLValueType.BOOL, cameraNodeKey .. "#useWorldXZRotation", "Use world XZ rotation")
    schema:register(XMLValueType.BOOL, cameraNodeKey .. "#resetCameraOnVehicleSwitch", "Reset camera on vehicle switch")
    schema:register(XMLValueType.INT, cameraNodeKey .. "#suspensionNodeIndex", "Index of seat suspension node")
    schema:register(XMLValueType.BOOL, cameraNodeKey .. "#useDefaultPositionSmoothing", "Use default position smoothing parameters", false)
    schema:register(XMLValueType.FLOAT, cameraNodeKey .. "#positionSmoothingParameter", "Position smoothing parameter", "0.128 for indoor / 0.016 for outside")
    schema:register(XMLValueType.FLOAT, cameraNodeKey .. "#lookAtSmoothingParameter", "Look at smoothing parameter", "0.176 for indoor / 0.022 for outside")

    schema:setXMLSpecializationType()
end)

function EnterablePassengerExtension.new(customEnvironment, baseDirectory, customMt)
    local self = setmetatable({}, customMt or EnterablePassengerExtension_mt)

    self.customEnvironment = customEnvironment
    self.baseDirectory = baseDirectory

    self.globalVehicles = {}
    self.showPassengersInStore = false

    return self
end

function EnterablePassengerExtension:load()
    addConsoleCommand("gtxShowPassengersInStore", "Displays vehicle passengers in the store screen", "consoleCommandShowPassengersInStore", self)

    if self.developmentMode then
        addConsoleCommand("gtxReloadGlobalPassengerVehicleList", "Reloads the global vehicle list [SP Only]", "consoleCommandReloadGlobalVehicleList", self)
    end
end

function EnterablePassengerExtension:delete()
    self.globalVehicles = {}
    self.showPassengersInStore = false

    removeConsoleCommand("gtxShowPassengersInStore")

    if self.developmentMode then
        removeConsoleCommand("gtxReloadGlobalPassengerVehicleList")
    end
end

function EnterablePassengerExtension:loadGlobalVehicles(loadModFiles)
    local vehiclesLoaded, modVehiclesLoaded = 0, 0

    vehiclesLoaded = vehiclesLoaded + self:loadGlobalVehicleFromXML(self.baseDirectory .. "shared/globalBaseVehicles.xml", nil)
    modVehiclesLoaded = modVehiclesLoaded + self:loadGlobalVehicleFromXML(self.baseDirectory .. "shared/globalModVehicles.xml", nil)

    if loadModFiles then
        -- Allow other mods and maps to load global passengers from external XML files.
        -- Not really required as modders should be adding passengers, but nice option for personal servers etc that want passengers in vehicles without needing to edit vehicles directly.
        for _, mod in pairs (g_modManager:getActiveMods()) do
            if mod.modName ~= self.customEnvironment then
                local xmlFile = loadXMLFile("tempModDesc", mod.modFile)
                local xmlFilename = getXMLString(xmlFile, "modDesc.enterablePassengerExtension#filename")

                if xmlFilename ~= nil then
                    modVehiclesLoaded = modVehiclesLoaded + self:loadGlobalVehicleFromXML(mod.modDir .. xmlFilename, mod.modName)
                end

                delete(xmlFile)
            end
        end
    end

    print(string.format("  Info: [EnterablePassengerExtension] Loaded %d base game global passenger vehicles.", vehiclesLoaded))

    if modVehiclesLoaded > 0 then
        print(string.format("  Info: [EnterablePassengerExtension] Loaded %d mod global passenger vehicles.", modVehiclesLoaded))
    end
end

function EnterablePassengerExtension:loadGlobalVehicleFromXML(filename, customEnvironment)
    local numVehicles = 0
    local xmlFile = XMLFile.loadIfExists("globalPassengerVehiclesXML", filename, EnterablePassengerExtension.xmlSchema)

    if xmlFile ~= nil then
        xmlFile:iterate("enterablePassengerExtension.vehicle", function (index, key)
            local xmlFilename = xmlFile:getValue(key .. "#xmlFilename")

            if xmlFilename ~= nil then
                local prefix = xmlFile:getValue(key .. "#prefix")
                local validForLoad = true

                if prefix ~= nil then
                    if prefix == "$moddir$" then
                        xmlFilename = g_modsDirectory .. xmlFilename
                    else
                        xmlFilename = NetworkUtil.convertFromNetworkFilename(prefix .. xmlFilename)
                    end

                    local vehicleModName, _ = Utils.getModNameAndBaseDirectory(xmlFilename)

                    if not g_modIsLoaded[vehicleModName] then
                        validForLoad = false
                    end
                end

                if validForLoad and fileExists(xmlFilename) then
                    if self.globalVehicles[xmlFilename] == nil then
                        self.globalVehicles[xmlFilename] = {
                            customEnvironment = customEnvironment,
                            xmlFilename = filename,
                            index = index - 1
                        }

                        numVehicles = numVehicles + 1
                    end
                end
            else
                Logging.xmlWarning(xmlFile, "No 'xmlFilename' given at '%s'!", key)
            end
        end)

        xmlFile:delete()
    else
        if customEnvironment ~= nil or self.developmentMode then
            Logging.warning("Failed to load XML file with filename '%s'!", filename or "UNKNOWN")
        end
    end

    return numVehicles
end

function EnterablePassengerExtension:getGlobalXMLData(configFileName)
    local globalVehicle = self.globalVehicles[configFileName]

    if globalVehicle ~= nil then
        local xmlFile = XMLFile.loadIfExists("globalPassengerVehiclesXML", globalVehicle.xmlFilename, EnterablePassengerExtension.xmlSchema)

        if xmlFile ~= nil then
            local key = string.format("enterablePassengerExtension.vehicle(%d)", globalVehicle.index)

            local xmlFilename = xmlFile:getValue(key .. "#xmlFilename")
            local prefix = xmlFile:getValue(key .. "#prefix")

            if prefix ~= nil then
                if prefix == "$moddir$" then
                    xmlFilename = g_modsDirectory .. xmlFilename
                else
                    xmlFilename = NetworkUtil.convertFromNetworkFilename(prefix .. xmlFilename)
                end
            end

            if xmlFilename == configFileName then
                return xmlFile, key
            end

            local index = 0
            local validKey = false

            xmlFile:iterate("enterablePassengerExtension.vehicle", function (_, vehicleKey)
                local xmlFilename = xmlFile:getValue(vehicleKey .. "#xmlFilename")
                local prefix = xmlFile:getValue(vehicleKey .. "#prefix")

                if prefix ~= nil then
                    if prefix == "$moddir$" then
                        xmlFilename = g_modsDirectory .. xmlFilename
                    else
                        xmlFilename = NetworkUtil.convertFromNetworkFilename(prefix .. xmlFilename)
                    end
                end

                if xmlFilename == configFileName then
                    key = vehicleKey
                    validKey = true

                    globalVehicle.index = index

                    return false
                end

                index = index + 1
            end)

            if validKey then
                return xmlFile, key
            end

            xmlFile:delete()
        else
            self.globalVehicles[configFileName] = nil
        end
    end
end

function EnterablePassengerExtension:getSinglePlayerPassengersEnabled()
    return true
end

function EnterablePassengerExtension:consoleCommandShowPassengersInStore()
    if g_currentMission == nil or g_currentMission.missionDynamicInfo == nil then
        return "Command only available when game is loaded!"
    end

    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return "Not possible in multiplayer!"
    end

    self.showPassengersInStore = not self.showPassengersInStore

    return "Show Passengers in store = " .. tostring(self.showPassengersInStore)
end

function EnterablePassengerExtension:consoleCommandReloadGlobalVehicleList(loadModFiles)
    if g_currentMission == nil or g_currentMission.missionDynamicInfo == nil then
        return "Reloading only available when game is loaded!"
    end

    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return "Reloading not possible in multiplayer!"
    end

    self.globalVehicles = {} -- Clear the list
    self:loadGlobalVehicles(Utils.stringToBoolean(loadModFiles))
end
