ColorConfiguratorListDialog = {
	CONTROLS = {
		"list",
		"textElement",
		"okButton",
		"noItemsText",
		"removeTextButton",
		"listSlider",
		"removeButton",
		"multiFilter"
	},
	L10N_SYMBOL = {
		USER_COLORS = "colorConfigurator_ui_userColors",
		LAST_COLORS = "colorConfigurator_ui_lastColors",
		GAME_COLORS = "colorConfigurator_ui_gameColors",
		INFO_REMOVE_COLOR = "colorConfigurator_ui_removeColor",
		ADD_COLORS = "colorConfigurator_ui_addColors"
	},
	SECTION = {
		BASE = 1,
		TEMP = 2,
		BRAND = 3,
		ADD = 4,
	}
}

local ColorConfiguratorListDialog_mt = Class(ColorConfiguratorListDialog, MessageDialog)

function ColorConfiguratorListDialog.new(target, custom_mt, colorConfigurator)
	local self = MessageDialog.new(target, custom_mt or ColorConfiguratorListDialog_mt)

	self:registerControls(ColorConfiguratorListDialog.CONTROLS)

	self.colorConfigurator = colorConfigurator

	self.disableOpenSound = true
	self.disableSearchOnClose = false
	self.textSample = nil

	self.listSections = {}

	self.brandColors = {}
	self.addSections = {}

	self.wrapAround = false

	self.lastInputHelpMode = nil

	self.filterState = colorConfigurator:getSetting("filterState")

	return self
end

function ColorConfiguratorListDialog:onMapLoad()
	self.brandColors = self:loadBrandColors()
	self.addSections = self:loadAdditionalColors(Utils.getFilename("colors.xml", self.colorConfigurator.baseDirectory), false)

	self:updateListData()
end

function ColorConfiguratorListDialog:onGuiSetupFinished()
	ColorConfiguratorListDialog:superClass().onGuiSetupFinished(self)

	self.multiFilter:setTexts({
		g_i18n:getText("ui_off"),
		g_i18n:getText(ColorConfiguratorListDialog.L10N_SYMBOL.USER_COLORS),
		g_i18n:getText(ColorConfiguratorListDialog.L10N_SYMBOL.LAST_COLORS),
		g_i18n:getText(ColorConfiguratorListDialog.L10N_SYMBOL.GAME_COLORS),
		g_i18n:getText(ColorConfiguratorListDialog.L10N_SYMBOL.ADD_COLORS)
	})

	if self.multiFilter.state ~= self.filterState then
		self.multiFilter:setState(self.filterState)
	end

	self.textElement.customFocusSample = self.textSample
	ColorConfiguratorUtil.overwrittenFunction(self.list, "shouldFocusChange", self, "list_shouldFocusChange")
end

function ColorConfiguratorListDialog:onClose()
	if self.disableSearchOnClose then
		self.textElement:setText("")
		self:onTextChanged(self.textElement, "")
	end

	self.colorConfigurator:updateSetting("filterState", self.filterState)

	ColorConfiguratorListDialog:superClass().onClose(self)
end

function ColorConfiguratorListDialog:onClickFilter(state)
	self.filterState = state

	self:updateListData()
end

function ColorConfiguratorListDialog:updateListData()
	self.listSections = {}

	local baseColors = {}
	local tempColors = {}
	local brandColors = {}
	local addSections = {}

	if self.filterState == 1 then
		baseColors = self.colorConfigurator:getBaseConfigs()
		tempColors = self.colorConfigurator:getTempConfigs()
		brandColors = self.brandColors
		addSections = self.addSections
	elseif self.filterState == 2 then
		baseColors = self.colorConfigurator:getBaseConfigs()
	elseif self.filterState == 3 then
		tempColors = self.colorConfigurator:getTempConfigs()
	elseif self.filterState == 4 then
		brandColors = self.brandColors
	elseif self.filterState == 5 then
		addSections = self.addSections
	end

	local searchText = self.textElement.text

	if searchText ~= "" then
		searchText = string.gsub(utf8ToLower(searchText), " ", "")

		local foundBaseColors = {}
		local foundTempColors = {}
		local foundBrandColors = {}
		local foundAddSections = {}

		for _, baseColor in pairs(baseColors) do
			if self:getCanAddItem(baseColor.name, searchText) then
				table.insert(foundBaseColors, baseColor)
			end
		end

		for _, tempColor in pairs(tempColors) do
			if self:getCanAddItem(tempColor.name, searchText) then
				table.insert(foundTempColors, tempColor)
			end
		end

		for _, brandColor in pairs(brandColors) do
			if self:getCanAddItem(brandColor.name, searchText) then
				table.insert(foundBrandColors, brandColor)
			end
		end

		for _, section in pairs(addSections) do
			local colors = {}

			for _, color in pairs(section.colors) do
				if self:getCanAddItem(color.name, searchText) then
					table.insert(colors, color)
				end
			end

			if #colors > 0 then
				table.insert(foundAddSections, {name = section.name, colors = colors})
			end
		end

		baseColors = foundBaseColors
		tempColors = foundTempColors
		brandColors = foundBrandColors
		addSections = foundAddSections
	end

	if #baseColors > 0 then
		table.insert(self.listSections, {
			id = ColorConfiguratorListDialog.SECTION.BASE,
			title = g_i18n:getText(ColorConfiguratorListDialog.L10N_SYMBOL.USER_COLORS),
			colorData = baseColors
		})
	end

	if #tempColors > 0 then
		table.insert(self.listSections, {
			id = ColorConfiguratorListDialog.SECTION.TEMP,
			title = g_i18n:getText(ColorConfiguratorListDialog.L10N_SYMBOL.LAST_COLORS),
			colorData = tempColors
		})
	end

	if #brandColors > 0 then
		table.insert(self.listSections, {
			id = ColorConfiguratorListDialog.SECTION.BRAND,
			title = g_i18n:getText(ColorConfiguratorListDialog.L10N_SYMBOL.GAME_COLORS),
			colorData = brandColors
		})
	end

	for _, section in pairs(addSections) do
		if #section.colors > 0 then
			table.insert(self.listSections, {
				id = ColorConfiguratorListDialog.SECTION.ADD,
				title = section.name,
				colorData = section.colors
			})
		end
	end

	self.list:reloadData()
	self:updateListElements()
end

function ColorConfiguratorListDialog:getNumberOfSections()
	return #self.listSections
end

function ColorConfiguratorListDialog:getTitleForSectionHeader(list, sectionIndex)
	return string.format("%s (%s)", self.listSections[sectionIndex].title, list.sections[sectionIndex].numItems)
end

function ColorConfiguratorListDialog:getNumberOfItemsInSection(list, sectionIndex)
	return #self.listSections[sectionIndex].colorData
end

function ColorConfiguratorListDialog:populateCellForItemInSection(list, sectionIndex, index, cell)
	local section = self.listSections[sectionIndex]
	local id = section.id
	local color = section.colorData[index]
	local name = color.name

	if id == ColorConfiguratorListDialog.SECTION.TEMP then
		name = string.format("%s, %s", color.date, color.name)
	end

	local r, g, b = unpack(color.color)
	local mat = Utils.getNoNil(color.material, 0)
	local isMetallic = ConfigurationUtil.isColorMetallic(mat)
	local isButtonVisible = id == ColorConfiguratorListDialog.SECTION.BASE

	cell:getAttribute("title"):setText(name)
	cell:getAttribute("image"):setImageColor(nil, r, g, b, 1)
	cell:getAttribute("image"):setVisible(not isMetallic)
	cell:getAttribute("imageMetallic"):setImageColor(nil, r, g, b, 1)
	cell:getAttribute("imageMetallic"):setVisible(isMetallic)
	cell:getAttribute("button"):setVisible(isButtonVisible)
end

function ColorConfiguratorListDialog:onClickColor(list, sectionIndex, index)
	if self.list.totalItemCount > 0 then
		local sectionIndex = self.list.selectedSectionIndex
		local index = self.list.selectedIndex

		local section = self.listSections[sectionIndex]
		local color = section.colorData[index]
		local r, g, b = unpack(color.color)
		local mat = color.material

		self.colorConfigurator.dialog:setValues({r, g, b, Utils.getNoNil(mat, 0)}, mat ~= nil)
	end

	self:playSample(GuiSoundPlayer.SOUND_SAMPLES.CLICK)
	self:close()
end

function ColorConfiguratorListDialog:onClickRemoveColor(element, useSelected)
	local listItem = nil

	if element == nil and useSelected then
		listItem = self.list:getSelectedElement()
	else
		listItem = element.parent.parent
	end

	if listItem ~= nil then
		local sectionIndex = listItem.sectionIndex
		local index = listItem.indexInSection

		local section = self.listSections[sectionIndex]
		local color = section.colorData[index]

		g_gui:showYesNoDialog({
			text = string.format(g_i18n:getText(ColorConfiguratorListDialog.L10N_SYMBOL.INFO_REMOVE_COLOR), color.name),
			callback = function (yes)
				if yes then
					if self.colorConfigurator:removeBaseConfig(color) then
						self:updateListData()
					end
				end
			end
		})
	end
end

function ColorConfiguratorListDialog:onClickRemoveText(element)
	self.textElement:setText("")
	self.textElement:setForcePressed(false)
	self:onTextChanged(self.textElement, "")
end

function ColorConfiguratorListDialog:onTextChanged(element, text)
	self:updateListData()
end

function ColorConfiguratorListDialog:updateListElements()
	local itemCount = self.list.totalItemCount
	local text = self.textElement.text

	self.removeTextButton:setVisible(text ~= "")
	self.noItemsText:setVisible(itemCount == 0)
	self.okButton:setDisabled(itemCount == 0)

	if itemCount == 0 then
		self.listSlider:setVisible(false)
	end
end

function ColorConfiguratorListDialog:getCanAddItem(itemText, searchText)
	itemText = string.gsub(utf8ToLower(itemText), " ", "")

	return string.find(itemText, searchText)
end

function ColorConfiguratorListDialog:loadBrandColors()
	local colorsList = {}

	for name, color in pairs(g_brandColorManager.brandColors) do
		local r, g, b, _ = unpack(color)

		table.insert(colorsList, {name = string.gsub(name, "_", " "), color = {r, g, b}})
	end

	table.sort(colorsList, function (a, b)
		return a.name < b.name
	end)

	return colorsList
end

function ColorConfiguratorListDialog:loadAdditionalColors(xmlFilename, printToLog)
	local sections = {}
	local xmlFile = XMLFile.load("colorsXML", xmlFilename)

	if xmlFile ~= nil then
		xmlFile:iterate("sections.section", function (_, sectionKey)
			local colors = {}

			xmlFile:iterate(sectionKey .. ".color", function (_, colorKey)
				local color = xmlFile:getVector(colorKey .. "#value")

				if color ~= nil then
					table.insert(colors, {name = xmlFile:getString(colorKey .. "#name", ""), color = color, material = color[4]})
				end
			end)

			table.sort(colors, function (a, b)
				return a.name < b.name
			end)

			local name = xmlFile:getString(sectionKey .. "#name", "")

			if name:sub(1, 6) == "$l10n_" then
				name = g_i18n:getText(name:sub(7))
			end

			table.insert(sections, {name = name, colors = colors})
		end)

		xmlFile:delete()

		ColorConfiguratorUtil.printOnLoad(printToLog, xmlFile)
	end

	return sections
end

function ColorConfiguratorListDialog:list_shouldFocusChange(list, superFunc, direction)
	local selectedIndex = list.selectedIndex
	local sectionIndex = list.selectedSectionIndex
	local retValue = superFunc(list, direction)

	if list.totalItemCount > 0 then
		local dir1 = FocusManager.TOP
		local dir2 = FocusManager.BOTTOM

		if direction == dir1 or direction == dir2 then
			local lastSectionIndex = #list.sections
			local lastIndex = list.sections[lastSectionIndex].numItems

			if direction == dir1 then
				if sectionIndex == 1 and selectedIndex == 1 then
					if self.wrapAround then
						list:setSelectedItem(lastSectionIndex, lastIndex, true, 0)
					end

					return false
				end
			else
				if sectionIndex == lastSectionIndex and selectedIndex == lastIndex then
					if self.wrapAround then
						list:setSelectedItem(1, 1, true, 0)
					end

					return false
				end
			end
		end
	end

	return retValue
end

function ColorConfiguratorListDialog:update(dt)
	ColorConfiguratorListDialog:superClass().update(self, dt)

	local inputHelpMode = g_inputBinding:getInputHelpMode()

	if inputHelpMode ~= self.lastInputHelpMode then
		self.lastInputHelpMode = inputHelpMode
		self:updateControllerButtons()
	end
end

function ColorConfiguratorListDialog:onListSelectionChanged(list, sectionIndex, index)
	self:updateControllerButtons()
end

function ColorConfiguratorListDialog:onClickRemove()
	self:onClickRemoveColor(nil, true)
end

function ColorConfiguratorListDialog:updateControllerButtons()
	self.removeButton:setVisible(self.lastInputHelpMode == GS_INPUT_HELP_MODE_GAMEPAD and self.listSections[self.list.selectedSectionIndex].id == ColorConfiguratorListDialog.SECTION.BASE)
end