ColorConfiguratorPickerDialog = {
	L10N_SYMBOL = {
		INFO_REMOVE_COLOR = "colorConfigurator_ui_removeColor",
		BUTTON_CUSTOM_COLOR = "colorConfigurator_button_customColor",
		BUTTON_LOAD = "button_load",
		BUTTON_DELETE = "button_delete"
	}
}

local ColorConfiguratorPickerDialog_mt = Class(ColorConfiguratorPickerDialog, ColorPickerDialog)

function ColorConfiguratorPickerDialog.new(target, custom_mt, colorConfigurator)
	local self = ColorPickerDialog.new(target, custom_mt or ColorConfiguratorPickerDialog_mt)

	self.colorConfigurator = colorConfigurator

	self.buttonPressedTimer = 0
	self.buttonPressedInterval = 250
	self.buttonPressed = nil

	self.openConfiguratorDialog = false
	self.removeButtons = {}
	self.removeButtonsMapping = {}
	self.previousColors = {}

	self.lastInputHelpMode = nil

	return self
end

function ColorConfiguratorPickerDialog:delete()
	if self.customizeButton ~= nil then
		self.customizeButton:delete()
	end

	if self.removeButtonBgTemplate ~= nil then
		self.removeButtonBgTemplate:delete()
	end

	if self.flowLayoutController ~= nil then
		self.flowLayoutController:delete()
	end

	ColorConfiguratorPickerDialog:superClass().delete(self)
end

function ColorConfiguratorPickerDialog:onGuiSetupFinished()
	ColorConfiguratorPickerDialog:superClass().onGuiSetupFinished(self)

	local buttonsPC = self.backButton.parent
	local customizeButton = ButtonElement.new(self)

	customizeButton:applyProfile("buttonActivate", true)
	customizeButton:setText(g_i18n:getText(ColorConfiguratorPickerDialog.L10N_SYMBOL.BUTTON_CUSTOM_COLOR))
	customizeButton:setCallback("onClickCallback", "onOpenConfiguratorDialog")
	customizeButton:applyScreenAlignment()

	table.insert(buttonsPC.elements, 2, customizeButton)
	customizeButton.parent = buttonsPC

	self.customizeButton = customizeButton

	local removeButton = ButtonElement.new(self)

	removeButton:applyProfile("colorConfiguratorRemoveButton", true)
	removeButton:setCallback("onClickCallback", "onClickRemoveColor")

	local removeButtonBgTemplate = BitmapElement.new(self)

	removeButtonBgTemplate:applyProfile("colorConfiguratorRemoveButtonBg", true)
	removeButtonBgTemplate:addElement(removeButton)
	removeButtonBgTemplate:applyScreenAlignment()

	self.dialogElement:addElement(removeButtonBgTemplate)

	self.removeButtonBgTemplate = removeButtonBgTemplate

	local xScale, yScale = removeButton:getAspectScale()
	local iconSizeX, iconSizeY = unpack(GuiUtils.getNormalizedValues("12px 12px", removeButton.outputSize))

	self.customConfigIconSize = {iconSizeX * xScale, iconSizeY * yScale}
	self.customConfigIconUVs = GuiUtils.getUVs("583px 7px 22px 22px")

	self.colorButtonLayout:setSize(unpack(GuiUtils.getNormalizedValues("416px 52px", self.colorButtonLayout.outputSize)))
	self.colorButtonLayout:applyScreenAlignment()

	local flowLayout = FlowLayoutElement.new(self)

	flowLayout:applyProfile("buttonBoxDialog", true)
	flowLayout:setPosition(unpack(GuiUtils.getNormalizedValues("0px -105px", flowLayout.outputSize, flowLayout.position)))
	flowLayout:applyScreenAlignment()

	self.dialogElement:addElement(flowLayout)

	local loadControllerButton = ButtonElement.new(self)

	loadControllerButton:applyProfile("buttonCancel", true)
	loadControllerButton:setText(g_i18n:getText(ColorConfiguratorPickerDialog.L10N_SYMBOL.BUTTON_LOAD))
	loadControllerButton:setCallback("onClickCallback", "onClickLoadColorController")

	local removeControllerButton = ButtonElement.new(self)

	removeControllerButton:applyProfile("buttonExtra2", true)
	removeControllerButton:setText(g_i18n:getText(ColorConfiguratorPickerDialog.L10N_SYMBOL.BUTTON_DELETE))
	removeControllerButton:setCallback("onClickCallback", "onClickRemoveColorController")

	flowLayout:addElement(loadControllerButton)
	flowLayout:addElement(removeControllerButton)

	self.loadControllerButton = loadControllerButton
	self.removeControllerButton = removeControllerButton
	self.flowLayoutController = flowLayout
end

function ColorConfiguratorPickerDialog:onOpenConfiguratorDialog(element, button)
	local dialog = g_gui:showDialog("ColorConfiguratorDialog", true)

	if dialog ~= nil and button ~= nil then
		local colorIndex = self.colorMapping[button]
		local config = self.colors[colorIndex]
		local color = config.color
		local mat = config.material

		dialog.target:setValues({color[1], color[2], color[3], Utils.getNoNil(mat, 0)}, mat ~= nil)
	end
end

function ColorConfiguratorPickerDialog:setColors(colors, defaultColor, defaultColorMaterial)
	self.previousColors = {colors, defaultColor, defaultColorMaterial}

	for i = #self.removeButtons, 1, -1 do
		self.removeButtons[i]:delete()
	end

	colors = table.copy(colors, math.huge)

	local customConfigIndex = nil
	local newColors = {}

	for i, color in ipairs(colors) do
		if color.isCustomConfig then
			if customConfigIndex == nil then
				customConfigIndex = i
			end
		else
			table.insert(newColors, color)
		end
	end

	local configName = self.args.configName
	local filename = self.target.storeItem.xmlFilename
	local addConfigs = self.colorConfigurator:getAdditionalConfigs(filename, configName)

	if addConfigs ~= nil then
		for _, addColor in ipairs(addConfigs) do
			table.insert(newColors, addColor)
		end
	end

	local colorIndex = self.target.configurations[configName]

	if colorIndex == customConfigIndex then
		if self.target.licensePlateData ~= nil and self.target.licensePlateData.colorConfiguratorData ~= nil then
			local customConfig = self.target.licensePlateData.colorConfiguratorData[configName]

			if customConfig ~= nil then
				local addCustomConfigButton = true
				local customConfigColor = customConfig.color
				local customConfigMaterial = customConfig.material

				if addConfigs ~= nil then
					for _, addConfig in ipairs(addConfigs) do
						local addConfigColor = addConfig.color
						local addConfigMaterial = addConfig.material

						if addConfigMaterial == customConfigMaterial and ColorConfiguratorUtil.equals(addConfigColor, customConfigColor) then
							addCustomConfigButton = false
							self.defaultColor = addConfig

							break
						end
					end
				end

				if addCustomConfigButton then
					local customColor = colors[customConfigIndex]

					customColor.color = customConfigColor
					customColor.material = customConfigMaterial

					table.insert(newColors, customColor)

					self.defaultColor = customColor
				end
			end
		end
	end

	ColorConfiguratorPickerDialog:superClass().setColors(self, newColors, defaultColor, defaultColorMaterial)

	self.removeButtons = {}
	self.removeButtonsMapping = {}

	for i, color in ipairs(newColors) do
		if color.isSelectable ~= false and color.isAddConfig or color.isCustomConfig then
			local removeButtonBg = self.removeButtonBgTemplate:clone(self.dialogElement)

			local position = self.buttonMapping[color].absPosition
			local size = self.buttonMapping[color].absSize
			local baseSize = removeButtonBg.absSize

			local x, y = position[1] + size[1] - baseSize[1], position[2] + size[2] - baseSize[2]

			removeButtonBg:setAbsolutePosition(x, y)
			removeButtonBg:setVisible(true)

			local removeButton = removeButtonBg.elements[1]

			if color.isCustomConfig then
				removeButton:setIconSize(unpack(self.customConfigIconSize))
				removeButton:setImageUVs(nil, self.customConfigIconUVs)
				removeButton:setDisabled(true)
			end

			table.insert(self.removeButtons, removeButtonBg)
			self.removeButtonsMapping[removeButton] = i
		end
	end

	self:updateBlurArea()
end

function ColorConfiguratorPickerDialog:onClickColorButton(element)
	local colorIndex = self.colorMapping[element]
	local selectedColor = self.colors[colorIndex]

	if selectedColor ~= nil and selectedColor.isAddConfig or selectedColor.isCustomConfig then
		self:onClickBack()

		local color = selectedColor.color
		local material = selectedColor.material
		local colors = {color[1], color[2], color[3], material}

		self.colorConfigurator:setColor(colors)
	else
		self:sendCallback(colorIndex)
	end
end

function ColorConfiguratorPickerDialog:update(dt)
	ColorConfiguratorPickerDialog:superClass().update(self, dt)

	local inputHelpMode = g_inputBinding:getInputHelpMode()

	if inputHelpMode ~= self.lastInputHelpMode then
		self.lastInputHelpMode = inputHelpMode
		self:updateControllerButtons()
	end

	if self.colors ~= nil then
		for i, color in ipairs(self.colors) do
			local button = self.buttonMapping[color]

			if button ~= nil then
				if button:getOverlayState() == GuiOverlay.STATE_PRESSED then
					self.buttonPressed = button
				end
			end
		end

		if self.buttonPressed ~= nil then
			if self.buttonPressed:getOverlayState() == GuiOverlay.STATE_PRESSED then
				self.buttonPressedTimer = self.buttonPressedTimer + dt

				if self.buttonPressedTimer > self.buttonPressedInterval then
					if not self.openConfiguratorDialog then
						self:onOpenConfiguratorDialog(nil, self.buttonPressed)
						self.openConfiguratorDialog = true
					end
				end
			else
				self.buttonPressedTimer = 0
				self.openConfiguratorDialog = false
				self.buttonPressed = nil
			end
		end
	end
end

function ColorConfiguratorPickerDialog:onClickRemoveColor(element, useSelected)
	local colorIndex = nil

	if element == nil and useSelected then
		colorIndex = self.colorMapping[self.currentSelectedElement]
	else
		colorIndex = self.removeButtonsMapping[element]
	end

	if colorIndex ~= nil then
		local selectedColor = self.colors[colorIndex]

		if selectedColor ~= nil and selectedColor.isAddConfig then
			g_gui:showYesNoDialog({
				text = string.format(g_i18n:getText(ColorConfiguratorPickerDialog.L10N_SYMBOL.INFO_REMOVE_COLOR), selectedColor.name),
				callback = function (yes)
					if yes then
						local storeItem = self.target.storeItem
						local filename = storeItem.xmlFilename
						local configName = self.args.configName

						if storeItem.bundleInfo ~= nil then
							for _, bundleItem in ipairs(storeItem.bundleInfo.bundleItems) do
								self.colorConfigurator:removeAdditionalConfig(bundleItem.xmlFilename, configName, selectedColor, true)
							end
						end

						if self.colorConfigurator:removeAdditionalConfig(filename, configName, selectedColor) then
							self:setColors(unpack(self.previousColors))
						end
					end
				end
			})
		end
	end
end

function ColorConfiguratorPickerDialog:updateBlurArea()
	if g_gui.guis[self.name].blurAreaActive then
		local x, y, width, height = self:getBlurArea()

		if x ~= nil then
			g_depthOfFieldManager:popArea()
			g_depthOfFieldManager:pushArea(x, y, width, height)
		end
	end
end

function ColorConfiguratorPickerDialog:onClickLoadColorController(element)
	self:onOpenConfiguratorDialog(nil, self.currentSelectedElement)
end

function ColorConfiguratorPickerDialog:onClickRemoveColorController(element)
	self:onClickRemoveColor(nil, true)
end

function ColorConfiguratorPickerDialog:onFocusColorButton(element)
	ColorConfiguratorPickerDialog:superClass().onFocusColorButton(self, element)
	self:updateControllerButtons()
end

function ColorConfiguratorPickerDialog:updateControllerButtons()
	local colorIndex = self.colorMapping[self.currentSelectedElement]
	local selectedColor = self.colors[colorIndex]

	self.loadControllerButton:setVisible(self.lastInputHelpMode == GS_INPUT_HELP_MODE_GAMEPAD)
	self.removeControllerButton:setVisible(self.lastInputHelpMode == GS_INPUT_HELP_MODE_GAMEPAD and selectedColor ~= nil and selectedColor.isAddConfig)
end