VehicleColor = {
	specName = string.format("%s.vehicleColor", g_currentModName),
	specEntryName = string.format("spec_%s.vehicleColor", g_currentModName)
}

function VehicleColor.prerequisitesPresent(specializations)
	return true
end

function VehicleColor.initSpecialization()
	local schemaSavegame = Vehicle.xmlSchemaSavegame

	schemaSavegame:register(XMLValueType.STRING, string.format("vehicles.vehicle(?).%s.configuration(?)#name", VehicleColor.specName), "Configuration name")
	schemaSavegame:register(XMLValueType.VECTOR_3, string.format("vehicles.vehicle(?).%s.configuration(?)#color", VehicleColor.specName), "Configuration color")
	schemaSavegame:register(XMLValueType.INT, string.format("vehicles.vehicle(?).%s.configuration(?)#material", VehicleColor.specName), "Configuration material")
end

function VehicleColor.registerFunctions(vehicleType)
	SpecializationUtil.registerFunction(vehicleType, "setCustomConfigurationsData", VehicleColor.setCustomConfigurationsData)
	SpecializationUtil.registerFunction(vehicleType, "getCustomConfigurations", VehicleColor.getCustomConfigurations)
	SpecializationUtil.registerFunction(vehicleType, "getCustomConfigurationByName", VehicleColor.getCustomConfigurationByName)
	SpecializationUtil.registerFunction(vehicleType, "getCustomConfigurationsDataIsEqual", VehicleColor.getCustomConfigurationsDataIsEqual)
	SpecializationUtil.registerFunction(vehicleType, "updateColorConfigurations", VehicleColor.updateColorConfigurations)
end

function VehicleColor.registerEventListeners(vehicleType)
	SpecializationUtil.registerEventListener(vehicleType, "onPreLoad", VehicleColor)
	SpecializationUtil.registerEventListener(vehicleType, "onReadStream", VehicleColor)
	SpecializationUtil.registerEventListener(vehicleType, "onWriteStream", VehicleColor)
end

function VehicleColor:onPreLoad(savegame)
	local spec = self[VehicleColor.specEntryName]

	spec.configs = {}

	if savegame ~= nil then
		local i = 0

		while true do
			local key = string.format("%s.%s.configuration(%d)", savegame.key, VehicleColor.specName, i)

			if not savegame.xmlFile:hasProperty(key) then
				break
			end

			local name = savegame.xmlFile:getValue(key .. "#name")
			local r, g, b = savegame.xmlFile:getValue(key .. "#color")
			local material = savegame.xmlFile:getValue(key .. "#material")

			spec.configs[name] = {
				color = {r, g, b},
				material = material
			}

			i = i + 1
		end
	end
end

function VehicleColor:saveToXMLFile(xmlFile, key, usedModNames)
	local spec = self[VehicleColor.specEntryName]
	local i = 0

	for configName, configData in pairs(spec.configs) do
		local configKey = string.format("%s.configuration(%d)", key, i)

		xmlFile:setValue(configKey .. "#name", configName)
		xmlFile:setValue(configKey .. "#color", configData.color[1], configData.color[2], configData.color[3])
		xmlFile:setValue(configKey .. "#material", configData.material)

		i = i + 1
	end
end

function VehicleColor:setCustomConfigurationsData(colorConfiguratorData)
	if colorConfiguratorData ~= nil then
		local spec = self[VehicleColor.specEntryName]
		local customConfigActive = false

		spec.configs = {}

		for configName, configData in pairs(colorConfiguratorData) do
			spec.configs[configName] = configData

			if configName == "baseMaterial" or string.startsWith(configName, "designMaterial") then
				self:applyBaseMaterialConfiguration(self.xmlFile, configName, self.configurations[configName])
			end

			ConfigurationUtil.applyConfigMaterials(self, self.xmlFile, configName, self.configurations[configName])

			if configName == "baseColor" or configName == "designColor" then
				ConfigurationUtil.setColor(self, self.xmlFile, configName, self.configurations[configName])
			end

			if configName == "wrappingColor" then
				self:applyBaseMaterialConfiguration(self.xmlFile, configName, self.configurations[configName])
			end

			customConfigActive = true
		end

		if customConfigActive then
			self:updateColorConfigurations()

			if self.applyBaseMaterial ~= nil then
				self:applyBaseMaterial()
			end
		end
	end
end

function VehicleColor:getCustomConfigurations()
	return self[VehicleColor.specEntryName].configs
end

function VehicleColor:getCustomConfigurationByName(configName)
	local configs = self:getCustomConfigurations()

	if configs ~= nil then
		return configs[configName]
	end

	return nil
end

function VehicleColor:onReadStream(streamId, connection)
	if connection:getIsServer() then
		local spec = self[VehicleColor.specEntryName]
		local data = {}
		local numConfigs = streamReadUInt8(streamId)

		for _ = 1, numConfigs do
			local configName = streamReadString(streamId)
			local r = streamReadFloat32(streamId)
			local g = streamReadFloat32(streamId)
			local b = streamReadFloat32(streamId)
			local material = streamReadInt32(streamId)

			data[configName] = {
				color = ColorConfiguratorUtil.formatColor(r, g, b),
				material = material
			}
		end

		self:setCustomConfigurationsData(data)
	end
end

function VehicleColor:onWriteStream(streamId, connection)
	if not connection:getIsServer() then
		local spec = self[VehicleColor.specEntryName]
		local numConfigs = 0

		for _, _ in pairs(spec.configs) do
			numConfigs = numConfigs + 1
		end

		streamWriteUInt8(streamId, numConfigs)

		for configName, configData in pairs(spec.configs) do
			streamWriteString(streamId, configName)
			streamWriteFloat32(streamId, configData.color[1])
			streamWriteFloat32(streamId, configData.color[2])
			streamWriteFloat32(streamId, configData.color[3])
			streamWriteInt32(streamId, configData.material)
		end
	end
end

function VehicleColor:getCustomConfigurationsDataIsEqual(data)
	if data ~= nil then
		for configName, configData in pairs(data) do
			local ownData = self:getCustomConfigurationByName(configName)

			if ownData ~= nil then
				if ownData.material ~= configData.material or not ColorConfiguratorUtil.equals(ownData.color, configData.color) then
					return false
				end
			end
		end
	end

	return true
end

function VehicleColor:updateColorConfigurations()
	VehicleColor.updateWheelsConfiguration(self)
	VehicleColor.updateCrawlersConfiguration(self)
	VehicleColor.updateUmbilicalReelConfiguration(self)
	VehicleColor.updateMultipleItemPurchaseConfiguration(self)
end

function VehicleColor:updateWheelsConfiguration()
	local spec_wheels = self.spec_wheels

	if spec_wheels ~= nil then
		spec_wheels.rimColor = self.xmlFile:getValue("vehicle.wheels.rimColor", nil, true)

		if spec_wheels.rimColor == nil and self.xmlFile:getValue("vehicle.wheels.rimColor#useBaseColor") then
			spec_wheels.rimColor = ConfigurationUtil.getColorByConfigId(self, "baseColor", self.configurations.baseColor) or ConfigurationUtil.getColorByConfigId(self, "baseMaterial", self.configurations.baseMaterial)
		end

		if spec_wheels.rimColor ~= nil then
			spec_wheels.rimColor[4] = self.xmlFile:getValue("vehicle.wheels.rimColor#material")
		end

		spec_wheels.overwrittenWheelColors = {
			_rimColor = {}
		}

		for i = 1, 8 do
			spec_wheels.overwrittenWheelColors[string.format("_hubColor%d", i - 1)] = {}
		end

		ConfigurationUtil.getOverwrittenMaterialColors(self, self.xmlFile, spec_wheels.overwrittenWheelColors)

		local overwrittenRimColor = spec_wheels.overwrittenWheelColors._rimColor

		if #overwrittenRimColor > 0 then
			if spec_wheels.rimColor == nil then
				spec_wheels.rimColor = overwrittenRimColor
			else
				spec_wheels.rimColor[3] = overwrittenRimColor[3]
				spec_wheels.rimColor[2] = overwrittenRimColor[2]
				spec_wheels.rimColor[1] = overwrittenRimColor[1]

				if #overwrittenRimColor == 4 then
					spec_wheels.rimColor[4] = overwrittenRimColor[4]
				end
			end
		end

		for j = 0, 7 do
			local hubsColorsKey = string.format("vehicle.wheels.hubs.color%d", j)
			local color = self.xmlFile:getValue(hubsColorsKey, nil, true)
			local material = self.xmlFile:getValue(hubsColorsKey .. "#material")

			if color ~= nil then
				spec_wheels.hubsColors[j] = color
				spec_wheels.hubsColors[j][4] = material
			elseif self.xmlFile:getValue(hubsColorsKey .. "#useBaseColor") then
				spec_wheels.hubsColors[j] = ConfigurationUtil.getColorByConfigId(self, "baseColor", self.configurations.baseColor) or ConfigurationUtil.getColorByConfigId(self, "baseMaterial", self.configurations.baseMaterial)
			elseif self.xmlFile:getValue(hubsColorsKey .. "#useRimColor") then
				spec_wheels.hubsColors[j] = Utils.getNoNil(ConfigurationUtil.getColorByConfigId(self, "rimColor", self.configurations.rimColor), spec_wheels.rimColor)
			end

			local overwrittenHubColor = spec_wheels.overwrittenWheelColors[string.format("_hubColor%d", j)]

			if #overwrittenHubColor > 0 then
				spec_wheels.hubsColors[j] = spec_wheels.hubsColors[j] or {
					0,
					0,
					0,
					0
				}
				spec_wheels.hubsColors[j][3] = overwrittenHubColor[3]
				spec_wheels.hubsColors[j][2] = overwrittenHubColor[2]
				spec_wheels.hubsColors[j][1] = overwrittenHubColor[1]

				if #overwrittenHubColor == 4 then
					spec_wheels.hubsColors[j][4] = overwrittenHubColor[4]
				end
			end

			for i, hub in pairs(spec_wheels.hubs) do
				local key = string.format("vehicle.wheels.hubs.hub(%d)", i - 1)
				local color = XMLUtil.getXMLOverwrittenValue(self.xmlFile, key, string.format(".color%d", j), "", "global")
				local material = XMLUtil.getXMLOverwrittenValue(self.xmlFile, key, string.format(".color%d#material", j), "")

				if color == "global" then
					color = spec_wheels.hubsColors[j]
				else
					color = ConfigurationUtil.getColorFromString(color)

					if color ~= nil then
						color[4] = material
					end
				end

				color = color or hub.colors[j]

				if color ~= nil then
					local r, g, b, mat = unpack(color)
					local _ = nil

					if mat == nil then
						_, _, _, mat = I3DUtil.getShaderParameterRec(hub.node, string.format("colorMat%d", j))
					end

					I3DUtil.setShaderParameterRec(hub.node, string.format("colorMat%d", j), r, g, b, mat, false)
				end
			end
		end

		local function applyWheelsColor(wheel, color, parentWheel)
			if color ~= nil then
				local r, g, b, mat, _ = unpack(color)
				mat = wheel.material or mat

				if wheel.wheelOuterRim ~= nil then
					if mat == nil then
						_, _, _, mat = I3DUtil.getShaderParameterRec(wheel.wheelOuterRim, "colorMat0")
					end

					I3DUtil.setShaderParameterRec(wheel.wheelOuterRim, "colorMat0", r, g, b, mat, false)
				end

				if wheel.wheelInnerRim ~= nil then
					if mat == nil then
						_, _, _, mat = I3DUtil.getShaderParameterRec(wheel.wheelInnerRim, "colorMat0")
					end

					I3DUtil.setShaderParameterRec(wheel.wheelInnerRim, "colorMat0", r, g, b, mat, false)
				end
			end

			if wheel.wheelInnerRim ~= nil then
				for i = 1, 7 do
					local color = spec_wheels.hubsColors[i]

					if color ~= nil then
						I3DUtil.setShaderParameterRec(wheel.wheelInnerRim, string.format("colorMat%d", i), color[1], color[2], color[3], color[4], false)
					end
				end
			end

			local additionalColor = Utils.getNoNil(wheel.additionalColor or parentWheel ~= nil and parentWheel.additionalColor or nil, color)

			if wheel.wheelAdditional ~= nil and additionalColor ~= nil then
				local r, g, b, _ = unpack(additionalColor)
				local _, _, _, w = I3DUtil.getShaderParameterRec(wheel.wheelAdditional, "colorMat0")

				if not wheel.additionalMaterial and parentWheel ~= nil then
					w = parentWheel.additionalMaterial or w
				end

				I3DUtil.setShaderParameterRec(wheel.wheelAdditional, "colorMat0", r, g, b, w, false)
			end

			if wheel.additionalWheels ~= nil then
				for _, additionalWheel in pairs(wheel.additionalWheels) do
					if additionalWheel.connector ~= nil then
						additionalWheel.connector.color = self:getWheelConfigurationValue(self.xmlFile, wheel.configIndex, string.split(additionalWheel.key, additionalWheel.singleKey)[1], additionalWheel.singleKey .. ".connector#color", nil, true)

						if additionalWheel.connector.color == nil then
							additionalWheel.connector.color = ConfigurationUtil.getColorByConfigId(self, "rimColor", self.configurations.rimColor) or wheel.color or spec_wheels.rimColor

						else
							additionalWheel.connector.color[4] = nil
						end

						if additionalWheel.connector.color ~= nil and getHasShaderParameter(additionalWheel.connector.node, "colorMat0") then
							local r, g, b, mat1 = unpack(additionalWheel.connector.color)
							local _, _, _, mat2 = I3DUtil.getShaderParameterRec(additionalWheel.connector.node, "colorMat0")

							I3DUtil.setShaderParameterRec(additionalWheel.connector.node, "colorMat0", r, g, b, mat1 or mat2, false)
						end
					end

					applyWheelsColor(additionalWheel, color, wheel)
				end
			end
		end

		local rimConfigColor = ConfigurationUtil.getColorByConfigId(self, "rimColor", self.configurations.rimColor)

		for _, wheel in pairs(spec_wheels.wheels) do
			applyWheelsColor(wheel, wheel.color or rimConfigColor or spec_wheels.rimColor)
		end

		for _, wheel in pairs(spec_wheels.dynamicallyLoadedWheels) do
			applyWheelsColor(wheel, wheel.color or rimConfigColor or spec_wheels.rimColor)
		end
	end
end

function VehicleColor:updateCrawlersConfiguration()
	local spec_crawlers = self.spec_crawlers
	local spec_wheels = self.spec_wheels

	if spec_crawlers ~= nil and spec_wheels ~= nil then
		local function applyCrawlersColor(xmlFile, crawler, color)
			local j = 0

			while true do
				local key = string.format("crawler.rimColorNodes.rimColorNode(%d)", j)

				if not xmlFile:hasProperty(key) then
					break
				end

				local node = xmlFile:getValue(key .. "#node", nil, crawler.loadedCrawler)

				if node ~= nil then
					local shaderParameter = xmlFile:getValue(key .. "#shaderParameter")

					if getHasShaderParameter(node, shaderParameter) then
						local r, g, b, mat = unpack(color)

						if mat == nil then
							local _ = nil
							_, _, _, mat = getShaderParameter(node, shaderParameter)
						end

						I3DUtil.setShaderParameterRec(node, shaderParameter, r, g, b, mat, true)
					end
				end

				j = j + 1
			end
		end

		local rimColor = Utils.getNoNil(ConfigurationUtil.getColorByConfigId(self, "rimColor", self.configurations.rimColor), spec_wheels.rimColor)

		if rimColor ~= nil then
			local wheelConfigId = Utils.getNoNil(self.configurations.wheel, 1)
			local wheelKey = string.format("vehicle.wheels.wheelConfigurations.wheelConfiguration(%d)", wheelConfigId - 1)
			local i = 1

			self.xmlFile:iterate(wheelKey .. ".crawlers.crawler", function (_, key)
				local xmlPath = self.xmlFile:getValue(key .. "#filename")
				local linkNode = self.xmlFile:getValue(key .. "#linkNode", nil, self.components, self.i3dMappings)

				if xmlPath ~= nil and linkNode ~= nil then
					local xmlFilename = Utils.getFilename(xmlPath, self.baseDirectory)
					local xmlFile = XMLFile.load("crawlerXml", xmlFilename, Crawlers.xmlSchema)

					if xmlFile ~= nil then
						local crawler = spec_crawlers.crawlers[i]

						if crawler ~= nil then
							crawler.rimColorNodes = applyCrawlersColor(xmlFile, crawler, rimColor)
						end

						i = i + 1

						xmlFile:delete()
					end
				end
			end)
		end
	end
end

function VehicleColor:updateUmbilicalReelConfiguration()
	local spec_umbilicalReel = self.spec_umbilicalReel

	if self.isClient and spec_umbilicalReel ~= nil and spec_umbilicalReel.reels ~= nil then
		local color = ConfigurationUtil.getColorByConfigId(self, "baseColor", self.configurations.baseColor)
		color[4] = color[4] or 0

		for _, reel in ipairs(spec_umbilicalReel.reels) do
			reel:setColor(color)
		end
	end
end

function VehicleColor:updateMultipleItemPurchaseConfiguration()
	local spec_multipleItemPurchase = self.spec_multipleItemPurchase

	if spec_multipleItemPurchase ~= nil and spec_multipleItemPurchase.loadedBales ~= nil then
		local color = ConfigurationUtil.getColorByConfigId(self, "baseColor", self.configurations.baseColor)

		for _, baleObject in ipairs(spec_multipleItemPurchase.loadedBales) do
			baleObject:setColor(unpack(color))
		end
	end
end