ColorConfiguratorUtil = {}

function ColorConfiguratorUtil.prependedFunction(oldTarget, oldFunc, newTarget, newFunc)
	local superFunc = oldTarget[oldFunc]

	oldTarget[oldFunc] = function(...)
		newTarget[newFunc](newTarget, ...)
		superFunc(...)
	end
end

function ColorConfiguratorUtil.appendedFunction(oldTarget, oldFunc, newTarget, newFunc)
	local superFunc = oldTarget[oldFunc]

	oldTarget[oldFunc] = function(...)
		superFunc(...)
		newTarget[newFunc](newTarget, ...)
	end
end

function ColorConfiguratorUtil.overwrittenFunction(oldTarget, oldFunc, newTarget, newFunc, isStatic)
	local superFunc = oldTarget[oldFunc]

	if isStatic then
		oldTarget[oldFunc] = function(...)
			return newTarget[newFunc](newTarget, superFunc, ...)
		end
	else
		oldTarget[oldFunc] = function(self, ...)
			return newTarget[newFunc](newTarget, self, superFunc, ...)
		end
	end
end

function ColorConfiguratorUtil.formatColor(r, g, b)
	local color = {}

	for _, value in pairs({r, g, b}) do
		table.insert(color, tonumber(string.format("%1.4f", value)))
	end

	return color
end

function ColorConfiguratorUtil.giantsToRgb(number, precision)
	number = MathUtil.clamp(tonumber(number) or 0, 0, 1)

	local sRGB = 0

	if number > 0.0031308 then
		sRGB = 1.055 * (math.pow(number, (1.0 / 2.4))) - 0.055
	else
		sRGB = 12.92 * number
	end

	return MathUtil.round(sRGB * 255, precision)
end

function ColorConfiguratorUtil.rgbToGiants(number, precision)
	number = MathUtil.clamp((tonumber(number) or 0) / 255, 0, 1)

	local linearSRGB = 0

	if number <= 0.04045 then
		linearSRGB = number / 12.92
	else
		linearSRGB = math.pow(((number + 0.055) / 1.055), 2.4)
	end

	return MathUtil.round(linearSRGB, precision or 4)
end

function ColorConfiguratorUtil.hueToRgb(hue, precision)
	local r, g, b = 0, 0, 0

	if hue >= 0 and hue < 60 then
		r, g, b = 1, hue / 60, 0
	elseif hue >= 60.0 and hue < 120 then
		r, g, b = 1 - (hue - 60) / 60, 1, 0
	elseif hue >= 120 and hue < 180 then
		r, g, b = 0, 1, (hue - 120) / 60
	elseif hue >= 180 and hue < 240 then
		r, g, b = 0, 1 - (hue - 180) / 60, 1
	elseif hue >= 240 and hue < 300 then
		r, g, b = (hue - 240) / 60, 0, 1
	elseif hue >= 300 and hue <= 360 then
		r, g, b = 1, 0, 1 - (hue - 300) / 60
	end

	return MathUtil.round(r * 255, precision), MathUtil.round(g * 255, precision), MathUtil.round(b * 255, precision)
end

function ColorConfiguratorUtil.rgbToHsv(r, g, b, precision)
	r, g, b = r / 255, g / 255, b / 255

	local cMax, cMin = math.max(r, g, b), math.min(r, g, b)
	local diff = cMax - cMin
	local h, s, v = 0, 0, cMax

	if cMax == 0 then
		s = 0
	else
		s = diff / cMax
	end

	if cMax == cMin then
		h = 0
	else
		if cMax == r then
			h = (g - b) / diff

			if g < b then
				h = h + 6
			end
		elseif cMax == g then
			h = (b - r) / diff + 2
		elseif cMax == b then
			h = (r - g) / diff + 4
		end

		h = h / 6
	end

	return MathUtil.round(h * 360, precision), MathUtil.round(s * 100, precision), MathUtil.round(v * 100, precision)
end

function ColorConfiguratorUtil.hsvToRgb(h, s, v, precision)
	h, s, v = h / 360, s / 100, v / 100

	local r, g, b = 0, 0, 0
	local i = math.floor(h * 6)
	local f = h * 6 - i
	local p = v * (1 - s)
	local q = v * (1 - f * s)
	local t = v * (1 - (1 - f) * s)

	i = i % 6

	if i == 0 then
		r, g, b = v, t, p
	elseif i == 1 then
		r, g, b = q, v, p
	elseif i == 2 then
		r, g, b = p, v, t
	elseif i == 3 then
		r, g, b = p, q, v
	elseif i == 4 then
		r, g, b = t, p, v
	elseif i == 5 then
		r, g, b = v, p, q
	end

	return MathUtil.round(r * 255, precision), MathUtil.round(g * 255, precision), MathUtil.round(b * 255, precision)
end

function ColorConfiguratorUtil.hsvToGiants(h, s, v)
	local r, g, b = ColorConfiguratorUtil.hsvToRgb(h, s, v)

	return ColorConfiguratorUtil.rgbToGiants(r), ColorConfiguratorUtil.rgbToGiants(g), ColorConfiguratorUtil.rgbToGiants(b)
end

function ColorConfiguratorUtil.equals(color1, color2)
	for i = 1, 3 do
		if ColorConfiguratorUtil.giantsToRgb(color1[i]) ~= ColorConfiguratorUtil.giantsToRgb(color2[i]) then
			return false
		end
	end

	return true
end

function ColorConfiguratorUtil.getClipboardText()
	local clipboardText = ""
	local unicode = nil
	local sym = Input.KEY_v
	local modifier = 4160
	local isDown = true
	local eventUsed = false
	local target = {
		blockTime = 0,
		getIsActive = function()
			return true
		end,
		getOverlayState = function()
			return GuiOverlay.STATE_PRESSED
		end,
		getText = function()
			return ""
		end,
		getIsUnicodeAllowed = function()
			return false
		end,
		updateVisibleTextElements = function()
		end,
		setText = function(_, text)
			clipboardText = string.split(text, "\n")[1]
		end
	}

	TextInputElement.keyEvent(target, unicode, sym, modifier, isDown, eventUsed)

	return clipboardText
end

function ColorConfiguratorUtil.printOnLoad(allow, xmlFile)
	if allow or g_isDevelopmentVersion then
		Logging.info("Loaded xml file '%s' from '%s'", xmlFile.objectName, xmlFile.filename)
	end
end

function ColorConfiguratorUtil.printOnSave(allow, xmlFile)
	if allow or g_isDevelopmentVersion then
		Logging.info("Saved xml file '%s' to '%s'", xmlFile.objectName, xmlFile.filename)
	end
end

function ColorConfiguratorUtil.convertFromNetworkFilename(filename)
	local filenameLower = filename:lower()
	local modPrefix = "$moddir$"
	local pdlcPrefix = "$pdlcdir"

	if string.startsWith(filenameLower, modPrefix) then
		local startIndex = modPrefix:len() + 1
		local modName = filename
		local f, l = filename:find("/", startIndex)

		if f ~= nil and l ~= nil and startIndex < f - 1 then
			modName = filename:sub(startIndex, f - 1)
		end

		local modDir = g_modNameToDirectory[modName]

		if modDir ~= nil then
			filename = modDir .. filename:sub(f + 1)
		end
	elseif string.startsWith(filenameLower, pdlcPrefix) then
		filename =  NetworkUtil.convertFromNetworkFilename(filename)
	end

	return filename
end