--[[
	RoundBalerExtension.lua
	
	Author: 	Ifko[nator]
	Date: 		24.03.2022
	Version: 	4.4
	
	Changelog:	v1.0 @24.12.2018 - initial implementation in FS 19
				---------------------------------------------------------------------------------------------------------------------------
				v1.5 @19.04.2020 - fix for straw harvest addon and the cruise control will not longer set to max, after bale unloading
				---------------------------------------------------------------------------------------------------------------------------
				v2.0 @12.08.2020 - added support for the fastBale from the Kverneland DLC. The Tractor will no longer stop with tihs baler.
				---------------------------------------------------------------------------------------------------------------------------
				v3.0 @28.09.2020 - added possibility to change the unload mode to manual. For fields with an big slope.
				---------------------------------------------------------------------------------------------------------------------------
				v4.0 @23.02.2021 - convert to FS 22
				---------------------------------------------------------------------------------------------------------------------------
				v4.1 @26.02.2022 - fix for square balers. They are no longer limited to 8 km/h if they had reached once 90% fill level
				---------------------------------------------------------------------------------------------------------------------------
				v4.2 @24.03.2022 - brakelights work now again, when baler is turned on
				---------------------------------------------------------------------------------------------------------------------------
				v4.3 @21.08.2022 - added support for vermeer DLC
				---------------------------------------------------------------------------------------------------------------------------
				v4.4 @24.03.2022 - added support for Hof Bergmann Version 1.2.0.0
]]

RoundBalerExtension = {};

function RoundBalerExtension.prerequisitesPresent(specializations)
	return SpecializationUtil.hasSpecialization(Drivable, specializations);
end;

function RoundBalerExtension.registerEventListeners(vehicleType)
	local functionNames = {
		"onLoad",
		"onUpdate"
	};
	
	for _, functionName in ipairs(functionNames) do
		SpecializationUtil.registerEventListener(vehicleType, functionName, RoundBalerExtension);
	end;
end;

function RoundBalerExtension.registerFunctions(vehicleType)
	local newFunctions = {
		"lockVehicle",
		"resetForNextBale",
		"setBaleToUnload",
		"handleVehicleTransmission",
		"getIsRoundBaler"
	};
	
	for _, newFunction in ipairs(newFunctions) do
		SpecializationUtil.registerFunction(vehicleType, newFunction, RoundBalerExtension[newFunction]);
	end;
end;

function RoundBalerExtension:onLoad(savegame)
	local specRoundBalerExtension = self.spec_roundBalerExtension;
	
	specRoundBalerExtension.hasBaleToUnload = false;
	specRoundBalerExtension.cruiseControlWasOn = false;
end;

function RoundBalerExtension:onUpdate(dt, isActiveForInput, isSelected)
	if self:getIsActive() then
		local specAttacherJoints = self.spec_attacherJoints;
		local specRoundBalerExtension = self.spec_roundBalerExtension;
		local specBaler = self.spec_baler;

		if specBaler ~= nil then
			local specDrivable = self.spec_drivable;
			local specWheels = self.spec_wheels;

			if self.speedLimitBackup == nil then
				self.speedLimitBackup = self.speedLimit;
			end;

			self:handleVehicleTransmission(specBaler, self, nil, specRoundBalerExtension, specDrivable, specWheels, nil);
		end;
		
		if specAttacherJoints ~= nil then
			for _, implement in pairs(specAttacherJoints.attachedImplements) do
				if implement ~= nil and implement.object ~= nil then
					local roundBaler = implement.object;
					local controlledTractor = roundBaler:getAttacherVehicle();

					local specDrivable = controlledTractor.spec_drivable;
					local specWheelsTractor = controlledTractor.spec_wheels;

					local specBaler = roundBaler.spec_baler;
					local specWheelsRoundBaler = roundBaler.spec_wheels;

					if roundBaler.speedLimitBackup == nil then
						roundBaler.speedLimitBackup = roundBaler.speedLimit;
					end;

					self:handleVehicleTransmission(specBaler, controlledTractor, roundBaler, specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler);
				end;
			end;
		end;
	end;
end;

function RoundBalerExtension:handleVehicleTransmission(specBaler, controlledTractor, roundBaler, specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler)
	roundBaler = Utils.getNoNil(roundBaler, controlledTractor);

	if self:getIsRoundBaler(specBaler, roundBaler) then
		--## we have an round baler :)
		
		if roundBaler:getFillUnitFillLevelPercentage(specBaler.fillUnitIndex) >= 0.9 and roundBaler:getFillUnitFreeCapacity(specBaler.fillUnitIndex) > 0 then
			roundBaler.speedLimit = 8;
		elseif roundBaler:getFillUnitFreeCapacity(specBaler.fillUnitIndex) == 0 then
			self:setBaleToUnload(specRoundBalerExtension, roundBaler);
			self:lockVehicle(specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, controlledTractor);
		elseif specBaler.unloadingState ~= Baler.UNLOADING_CLOSED then
			self:lockVehicle(specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, controlledTractor);
		elseif specBaler.unloadingState == Baler.UNLOADING_CLOSED then
			if specRoundBalerExtension.hasBaleToUnload then
				if #specBaler.bales == 0 then
					self:resetForNextBale(specRoundBalerExtension, controlledTractor);
				else
					self:lockVehicle(specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, controlledTractor);
				end;
			end;
		end;
	end;
end;

function RoundBalerExtension:getIsRoundBaler(specBaler, roundBaler)
	return specBaler ~= nil and specBaler.hasUnloadingAnimation and roundBaler.getIsTurnedOn ~= nil and roundBaler:getIsTurnedOn() and not string.find(roundBaler.configFileName, "fastBale");
end;

function RoundBalerExtension:resetForNextBale(specRoundBalerExtension, controlledTractor)
	specRoundBalerExtension.hasBaleToUnload = false;
	
	if specRoundBalerExtension.cruiseControlWasOn then	
		controlledTractor:setCruiseControlState(Drivable.CRUISECONTROL_STATE_ACTIVE);

		specRoundBalerExtension.cruiseControlWasOn = false;
	end;
end;

function RoundBalerExtension:lockVehicle(specRoundBalerExtension, specDrivable, specWheelsTractor, specWheelsRoundBaler, controlledTractor)
	if self.isServer then
		for _, wheel in pairs(specWheelsTractor.wheels) do
			setWheelShapeProps(wheel.node, wheel.wheelShape, 0, math.huge, 0, 0);
		end;
		
		if specWheelsRoundBaler ~= nil then
			for _, wheel in pairs(specWheelsRoundBaler.wheels) do
				setWheelShapeProps(wheel.node, wheel.wheelShape, 0, math.huge, 0, 0);
			end;
		end;
	end;

	controlledTractor:setBrakeLightsVisibility(true);

	if specDrivable.cruiseControl.state ~= Drivable.CRUISECONTROL_STATE_OFF then
		specRoundBalerExtension.cruiseControlWasOn = true;
		
		controlledTractor:setCruiseControlState(Drivable.CRUISECONTROL_STATE_OFF);
	end;
end;

function RoundBalerExtension:setBaleToUnload(specRoundBalerExtension, roundBaler)
	if not specRoundBalerExtension.hasBaleToUnload then
		specRoundBalerExtension.hasBaleToUnload = true;

		roundBaler.speedLimit = roundBaler.speedLimitBackup;
	end;
end;